<?php
// /stock/index.php

require_once '/home/crusardi/config_crusardi/database.php';
require_once __DIR__ . '/vendor/autoload.php';

// Sesión
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verificar autenticación
if (!isset($_SESSION['user_email'])) {
    // Si es una petición AJAX, devolver JSON en lugar de redireccionar
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
        strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        
        header('Content-Type: application/json; charset=utf-8');
        header('HTTP/1.1 401 Unauthorized');
        echo json_encode([
            'success' => false, 
            'message' => 'Sesión expirada. Por favor, recargue la página.',
            'redirect' => 'auth/login.php'
        ]);
        exit;
    }
    
    // Si no es AJAX, redireccionar normalmente
    header('Location: auth/login.php');
    exit;
}

// Página actual
$page = $_GET['page'] ?? 'stock_calculado';
$allowed = [
    'stock_calculado',
    'importar_stock',
    'importar_combinaciones',
    'importar_productos',
    'cargar_imagenes',
    'descuentos',
    'excepciones',
    'gestionar_excepciones_kits',
    'productos_imperfectos',
    'upload_foto_imperfectos',
    'usuarios',
    'gestionar_categorias'
];
if (!in_array($page, $allowed, true)) {
    $page = 'stock_calculado';
}

// FUNCIONES DE SINCRONIZACIÓN DE CATEGORÍAS
function mapear_categoria_db($pdo, $grupo_articulo) {
    $grupo_upper = strtoupper(trim($grupo_articulo));
    
    // Buscar en la tabla de mapeo actualizada
    $stmt = $pdo->prepare("
        SELECT 
            c1.nombre as cat1, 
            c2.nombre as cat2
        FROM categoria_mapping cm
        LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
        LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
        WHERE cm.grupo_articulo = ? AND cm.activo = 1
    ");
    $stmt->execute([$grupo_upper]);
    $result = $stmt->fetch();
    
    if ($result) {
        return ['cat1' => $result['cat1'] ?: 'Otros', 'cat2' => $result['cat2'] ?: 'General'];
    }
    
    // Si no hay coincidencia exacta, buscar coincidencia parcial
    $stmt = $pdo->prepare("
        SELECT 
            c1.nombre as cat1, 
            c2.nombre as cat2
        FROM categoria_mapping cm
        LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
        LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
        WHERE ? LIKE CONCAT('%', cm.grupo_articulo, '%') AND cm.activo = 1 
        ORDER BY LENGTH(cm.grupo_articulo) DESC LIMIT 1
    ");
    $stmt->execute([$grupo_upper]);
    $result = $stmt->fetch();
    
    if ($result) {
        return ['cat1' => $result['cat1'] ?: 'Otros', 'cat2' => $result['cat2'] ?: 'General'];
    }
    
    // Default
    return ['cat1' => 'Otros', 'cat2' => 'General'];
}

// Función para actualizar productos de un grupo específico cuando se cambia un mapeo
function actualizarProductosGrupo($pdo, $grupo_articulo) {
    try {
        // Obtener las nuevas categorías para este grupo
        $categorias = mapear_categoria_db($pdo, $grupo_articulo);
        
        // Actualizar todos los productos de este grupo
        $stmt = $pdo->prepare("
            UPDATE references_data 
            SET categoria1 = ?, categoria2 = ? 
            WHERE grupo_articulo = ?
        ");
        $stmt->execute([
            $categorias['cat1'],
            $categorias['cat2'],
            $grupo_articulo
        ]);
        
        return $stmt->rowCount();
    } catch (Exception $e) {
        throw new Exception("Error actualizando productos del grupo '$grupo_articulo': " . $e->getMessage());
    }
}

// Función para sincronizar todas las categorías de productos
function sincronizarCategoriasProductos($pdo) {
    $startTime = microtime(true);
    $log_messages = [];
    $productos_actualizados = 0;
    $productos_sin_mapeo = 0;
    
    try {
        $log_messages[] = "[INFO] " . date('H:i:s') . " - Iniciando sincronización de categorías de productos";
        
        // Obtener todos los productos que tienen grupo_articulo
        $stmt = $pdo->query("
            SELECT referencia, grupo_articulo 
            FROM references_data 
            WHERE grupo_articulo IS NOT NULL 
            AND grupo_articulo != ''
        ");
        $productos = $stmt->fetchAll();
        
        $total_productos = count($productos);
        $log_messages[] = "[INFO] " . date('H:i:s') . " - Encontrados $total_productos productos para sincronizar";
        
        // Preparar statement de actualización
        $update_stmt = $pdo->prepare("
            UPDATE references_data 
            SET categoria1 = ?, categoria2 = ? 
            WHERE referencia = ?
        ");
        
        foreach ($productos as $producto) {
            $referencia = $producto['referencia'];
            $grupo_articulo = $producto['grupo_articulo'];
            
            // Buscar mapeo para este grupo
            $categorias = mapear_categoria_db($pdo, $grupo_articulo);
            
            if ($categorias['cat1'] !== 'Otros' || $categorias['cat2'] !== 'General') {
                // Actualizar producto con las categorías mapeadas
                $update_stmt->execute([
                    $categorias['cat1'],
                    $categorias['cat2'],
                    $referencia
                ]);
                $productos_actualizados++;
                
                if ($productos_actualizados % 100 == 0) {
                    $log_messages[] = "[PROGRESS] " . date('H:i:s') . " - Actualizados $productos_actualizados productos...";
                }
            } else {
                $productos_sin_mapeo++;
            }
        }
        
        $endTime = microtime(true);
        $executionTime = round($endTime - $startTime, 2);
        
        $log_messages[] = "[SUCCESS] " . date('H:i:s') . " - Sincronización completada en {$executionTime}s";
        $log_messages[] = "[STATS] " . date('H:i:s') . " - Productos actualizados: $productos_actualizados";
        $log_messages[] = "[STATS] " . date('H:i:s') . " - Productos sin mapeo: $productos_sin_mapeo";
        
        return [
            'success' => true,
            'productos_actualizados' => $productos_actualizados,
            'productos_sin_mapeo' => $productos_sin_mapeo,
            'logs' => $log_messages
        ];
        
    } catch (Exception $e) {
        $log_messages[] = "[ERROR] " . date('H:i:s') . " - Error en sincronización: " . $e->getMessage();
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'logs' => $log_messages
        ];
    }
}

// FUNCIÓN PARA VERIFICAR ESTRUCTURA DE TABLAS
function verificarEstructuraTablas($pdo) {
    try {
        // Verificar si las tablas existen y tienen la estructura correcta
        $tablas_requeridas = [
            'grupos_axapta' => ['id', 'codigo', 'descripcion', 'activo'],
            'categorias_nivel1' => ['id', 'nombre', 'orden', 'activo'],
            'categorias_nivel2' => ['id', 'categoria1_id', 'nombre', 'orden', 'activo'],
            'categoria_mapping' => ['id', 'grupo_articulo', 'categoria1_id', 'categoria2_id', 'activo']
        ];
        
        foreach ($tablas_requeridas as $tabla => $columnas) {
            $stmt = $pdo->prepare("DESCRIBE `$tabla`");
            $stmt->execute();
            $estructura = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            foreach ($columnas as $columna) {
                if (!in_array($columna, $estructura)) {
                    throw new Exception("La tabla '$tabla' no tiene la columna '$columna'. Ejecuta el script SQL de corrección.");
                }
            }
        }
        return true;
    } catch (PDOException $e) {
        throw new Exception("Error verificando estructura de tablas: " . $e->getMessage());
    }
}

// PROCESAR ACCIONES POST ANTES DE CUALQUIER OUTPUT
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Para gestionar_categorias, procesar directamente aqui
    if ($page === 'gestionar_categorias') {
        $action = $_POST['action'] ?? '';
        $current_role = $_SESSION['user_role'];
        
        // Solo admin y editor pueden gestionar categorías
        if (!in_array($current_role, ['admin', 'editor'])) {
            header('Location: index.php');
            exit;
        }
        
        try {
            // Verificar estructura de tablas antes de procesar
            verificarEstructuraTablas($pdo);
            
            // Acciones para grupos de Axapta
            if ($action === 'create_grupo') {
                $codigo = strtoupper(trim($_POST['codigo']));
                $descripcion = trim($_POST['descripcion'] ?? '');
                
                // Verificar si el código ya existe
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM grupos_axapta WHERE codigo = ?");
                $stmt->execute([$codigo]);
                if ($stmt->fetchColumn() > 0) {
                    $_SESSION['error_message'] = "El código '$codigo' ya existe.";
                } else {
                    $stmt = $pdo->prepare("INSERT INTO grupos_axapta (codigo, descripcion) VALUES (?, ?)");
                    $stmt->execute([$codigo, $descripcion]);
                    $_SESSION['success_message'] = "Grupo de Axapta creado exitosamente.";
                }
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'grupos'));
                exit();
            }
            elseif ($action === 'update_grupo') {
                $stmt = $pdo->prepare("UPDATE grupos_axapta SET codigo = ?, descripcion = ?, activo = ? WHERE id = ?");
                $stmt->execute([
                    strtoupper(trim($_POST['codigo'])),
                    trim($_POST['descripcion'] ?? ''),
                    isset($_POST['activo']) ? 1 : 0,
                    $_POST['id']
                ]);
                $_SESSION['success_message'] = "Grupo de Axapta actualizado exitosamente.";
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'grupos'));
                exit();
            }
            elseif ($action === 'delete_grupo' && $current_role === 'admin') {
                // Verificar si el grupo está siendo usado en mapeos
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM categoria_mapping cm 
                                       JOIN grupos_axapta ga ON ga.codigo COLLATE utf8mb4_0900_ai_ci = cm.grupo_articulo COLLATE utf8mb4_0900_ai_ci 
                                       WHERE ga.id = ?");
                $stmt->execute([$_POST['id']]);
                $count = $stmt->fetchColumn();
                
                if ($count > 0) {
                    $_SESSION['error_message'] = "No se puede eliminar el grupo porque está siendo usado en mapeos.";
                } else {
                    $stmt = $pdo->prepare("DELETE FROM grupos_axapta WHERE id = ?");
                    $stmt->execute([$_POST['id']]);
                    $_SESSION['success_message'] = "Grupo de Axapta eliminado exitosamente.";
                }
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'grupos'));
                exit();
            }
            // Acciones para categorías nivel 1
            elseif ($action === 'create_cat1') {
                $nombre = trim($_POST['nombre']);
                $orden = (int)($_POST['orden'] ?? 99);
                
                // Verificar si el nombre ya existe
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM categorias_nivel1 WHERE nombre = ?");
                $stmt->execute([$nombre]);
                if ($stmt->fetchColumn() > 0) {
                    $_SESSION['error_message'] = "Ya existe una categoría con el nombre '$nombre'.";
                } else {
                    $stmt = $pdo->prepare("INSERT INTO categorias_nivel1 (nombre, orden) VALUES (?, ?)");
                    $stmt->execute([$nombre, $orden]);
                    $_SESSION['success_message'] = "Categoría principal creada exitosamente.";
                }
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'cat1'));
                exit();
            }
            elseif ($action === 'update_cat1') {
                $stmt = $pdo->prepare("UPDATE categorias_nivel1 SET nombre = ?, orden = ?, activo = ? WHERE id = ?");
                $stmt->execute([
                    trim($_POST['nombre']),
                    (int)($_POST['orden'] ?? 99),
                    isset($_POST['activo']) ? 1 : 0,
                    $_POST['id']
                ]);
                $_SESSION['success_message'] = "Categoría principal actualizada exitosamente.";
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'cat1'));
                exit();
            }
            elseif ($action === 'delete_cat1' && $current_role === 'admin') {
                // Verificar si tiene subcategorías
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM categorias_nivel2 WHERE categoria1_id = ?");
                $stmt->execute([$_POST['id']]);
                if ($stmt->fetchColumn() > 0) {
                    $_SESSION['error_message'] = "No se puede eliminar la categoría porque tiene subcategorías asociadas.";
                } else {
                    $stmt = $pdo->prepare("DELETE FROM categorias_nivel1 WHERE id = ?");
                    $stmt->execute([$_POST['id']]);
                    $_SESSION['success_message'] = "Categoría principal eliminada exitosamente.";
                }
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'cat1'));
                exit();
            }
            
            // Acciones para categorías nivel 2
            elseif ($action === 'create_cat2') {
                $categoria1_id = $_POST['categoria1_id'];
                $nombre = trim($_POST['nombre']);
                $orden = (int)($_POST['orden'] ?? 99);
                
                // Verificar si la categoría padre existe
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM categorias_nivel1 WHERE id = ?");
                $stmt->execute([$categoria1_id]);
                if ($stmt->fetchColumn() == 0) {
                    $_SESSION['error_message'] = "La categoría principal seleccionada no existe.";
                } else {
                    // Verificar si ya existe una subcategoría con el mismo nombre en la misma categoría padre
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM categorias_nivel2 WHERE categoria1_id = ? AND nombre = ?");
                    $stmt->execute([$categoria1_id, $nombre]);
                    if ($stmt->fetchColumn() > 0) {
                        $_SESSION['error_message'] = "Ya existe una subcategoría con ese nombre en esta categoría principal.";
                    } else {
                        $stmt = $pdo->prepare("INSERT INTO categorias_nivel2 (categoria1_id, nombre, orden) VALUES (?, ?, ?)");
                        $stmt->execute([$categoria1_id, $nombre, $orden]);
                        $_SESSION['success_message'] = "Subcategoría creada exitosamente.";
                    }
                }
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'cat2'));
                exit();
            }
            elseif ($action === 'update_cat2') {
                $stmt = $pdo->prepare("UPDATE categorias_nivel2 SET categoria1_id = ?, nombre = ?, orden = ?, activo = ? WHERE id = ?");
                $stmt->execute([
                    $_POST['categoria1_id'],
                    trim($_POST['nombre']),
                    (int)($_POST['orden'] ?? 99),
                    isset($_POST['activo']) ? 1 : 0,
                    $_POST['id']
                ]);
                $_SESSION['success_message'] = "Subcategoría actualizada exitosamente.";
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'cat2'));
                exit();
            }
            elseif ($action === 'delete_cat2' && $current_role === 'admin') {
                // Verificar si está siendo usada en mapeos
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM categoria_mapping WHERE categoria2_id = ?");
                $stmt->execute([$_POST['id']]);
                if ($stmt->fetchColumn() > 0) {
                    $_SESSION['error_message'] = "No se puede eliminar la subcategoría porque está siendo usada en mapeos.";
                } else {
                    $stmt = $pdo->prepare("DELETE FROM categorias_nivel2 WHERE id = ?");
                    $stmt->execute([$_POST['id']]);
                    $_SESSION['success_message'] = "Subcategoría eliminada exitosamente.";
                }
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'cat2'));
                exit();
            }
            
            // Acciones para mapeo
            elseif ($action === 'create_mapping') {
                $grupo_articulo = strtoupper(trim($_POST['grupo_articulo']));
                $categoria1_id = $_POST['categoria1_id'];
                $categoria2_id = $_POST['categoria2_id'] ?: null;
                
                // Verificar si el grupo ya está mapeado
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM categoria_mapping WHERE grupo_articulo = ?");
                $stmt->execute([$grupo_articulo]);
                if ($stmt->fetchColumn() > 0) {
                    $_SESSION['error_message'] = "El grupo '$grupo_articulo' ya está mapeado.";
                } else {
                    // Verificar que las categorías existen
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM categorias_nivel1 WHERE id = ?");
                    $stmt->execute([$categoria1_id]);
                    if ($stmt->fetchColumn() == 0) {
                        $_SESSION['error_message'] = "La categoría principal seleccionada no existe.";
                    } else {
                        // Si se seleccionó subcategoría, verificar que existe
                        if ($categoria2_id) {
                            $stmt = $pdo->prepare("SELECT COUNT(*) FROM categorias_nivel2 WHERE id = ? AND categoria1_id = ?");
                            $stmt->execute([$categoria2_id, $categoria1_id]);
                            if ($stmt->fetchColumn() == 0) {
                                $_SESSION['error_message'] = "La subcategoría seleccionada no existe o no pertenece a la categoría principal.";
                                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'mapeo'));
                                exit();
                            }
                        }
                        
                        $stmt = $pdo->prepare("INSERT INTO categoria_mapping (grupo_articulo, categoria1_id, categoria2_id, activo) VALUES (?, ?, ?, 1)");
                        $stmt->execute([$grupo_articulo, $categoria1_id, $categoria2_id]);
                        
                        // NUEVO: Actualizar productos con este grupo
                        try {
                            $productos_actualizados = actualizarProductosGrupo($pdo, $grupo_articulo);
                            $_SESSION['success_message'] = "Mapeo creado exitosamente. Se actualizaron $productos_actualizados productos.";
                        } catch (Exception $e) {
                            $_SESSION['success_message'] = "Mapeo creado exitosamente, pero error actualizando productos: " . $e->getMessage();
                        }
                    }
                }
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'mapeo'));
                exit();
            }
            elseif ($action === 'update_mapping') {
                $grupo_articulo = strtoupper(trim($_POST['grupo_articulo']));
                
                $stmt = $pdo->prepare("UPDATE categoria_mapping SET grupo_articulo = ?, categoria1_id = ?, categoria2_id = ?, activo = ? WHERE id = ?");
                $stmt->execute([
                    $grupo_articulo,
                    $_POST['categoria1_id'],
                    $_POST['categoria2_id'] ?: null,
                    isset($_POST['activo']) ? 1 : 0,
                    $_POST['id']
                ]);
                
                // NUEVO: Actualizar productos con este grupo
                try {
                    $productos_actualizados = actualizarProductosGrupo($pdo, $grupo_articulo);
                    $_SESSION['success_message'] = "Mapeo actualizado exitosamente. Se actualizaron $productos_actualizados productos.";
                } catch (Exception $e) {
                    $_SESSION['success_message'] = "Mapeo actualizado exitosamente, pero error actualizando productos: " . $e->getMessage();
                }
                
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'mapeo'));
                exit();
            }
            elseif ($action === 'delete_mapping' && $current_role === 'admin') {
                $stmt = $pdo->prepare("DELETE FROM categoria_mapping WHERE id = ?");
                $stmt->execute([$_POST['id']]);
                $_SESSION['success_message'] = "Mapeo eliminado exitosamente.";
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'mapeo'));
                exit();
            }
            
            // NUEVAS ACCIONES DE SINCRONIZACIÓN
            elseif ($action === 'sync_all_categories') {
                $resultado = sincronizarCategoriasProductos($pdo);
                
                if ($resultado['success']) {
                    $_SESSION['success_message'] = "Sincronización completada: {$resultado['productos_actualizados']} productos actualizados, {$resultado['productos_sin_mapeo']} sin mapeo.";
                    $_SESSION['sync_logs'] = $resultado['logs'];
                } else {
                    $_SESSION['error_message'] = "Error en sincronización: " . $resultado['error'];
                    $_SESSION['sync_logs'] = $resultado['logs'];
                }
                
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'mapeo'));
                exit();
            }
            elseif ($action === 'sync_group_products') {
                try {
                    $grupo_articulo = $_POST['grupo_articulo'];
                    $productos_actualizados = actualizarProductosGrupo($pdo, $grupo_articulo);
                    $_SESSION['success_message'] = "Grupo '$grupo_articulo' sincronizado: $productos_actualizados productos actualizados.";
                } catch (Exception $e) {
                    $_SESSION['error_message'] = "Error sincronizando grupo: " . $e->getMessage();
                }
                
                header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'mapeo'));
                exit();
            }
            
        } catch (Exception $e) {
            $_SESSION['error_message'] = "Error: " . $e->getMessage();
            header('Location: index.php?page=gestionar_categorias&tab=' . ($_POST['current_tab'] ?? 'mapeo'));
            exit();
        }
    }
}

// Gestión de acciones generales - ESTO DEBE IR ANTES DE CUALQUIER OUTPUT
if (isset($_REQUEST['action'])) {
    $action = basename($_REQUEST['action']);
    $action_file = __DIR__ . '/pages/actions/' . $action . '.php';
    if (file_exists($action_file)) {
        include $action_file;
        exit;
    }
}

$page_path = __DIR__ . '/pages/' . $page . '.php';

// AHORA SÍ EMPIEZA EL HTML
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>Stock BoConcept - Crusardi</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; }
        .sidebar-link {
            transition: all 0.2s ease;
        }
        .sidebar-link:hover {
            transform: translateX(4px);
        }
        @keyframes slideIn {
            from { transform: translateX(-100%); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
        .animate-slide-in {
            animation: slideIn 0.3s ease-out;
        }
    </style>
    <script src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js" defer></script>
</head>
<body class="bg-gray-50">

    <div x-data="{ sidebarOpen: false }" class="relative min-h-screen flex">

        <!-- Overlay móvil -->
        <div x-show="sidebarOpen" 
             x-transition:enter="transition-opacity ease-linear duration-300"
             x-transition:enter-start="opacity-0"
             x-transition:enter-end="opacity-100"
             x-transition:leave="transition-opacity ease-linear duration-300"
             x-transition:leave-start="opacity-100"
             x-transition:leave-end="opacity-0"
             @click="sidebarOpen = false"
             class="fixed inset-0 bg-black bg-opacity-50 z-30 lg:hidden"></div>

        <!-- Sidebar -->
        <aside 
            class="fixed inset-y-0 left-0 bg-gradient-to-b from-gray-900 to-gray-800 text-white w-72 py-6 px-5 transform transition-transform duration-300 ease-in-out z-40 lg:relative lg:translate-x-0 shadow-xl"
            :class="{'translate-x-0 animate-slide-in': sidebarOpen, '-translate-x-full': !sidebarOpen}">
            
            <!-- Logo y título -->
            <div class="flex flex-col items-center mb-8 pb-6 border-b border-gray-700">
                <div class="bg-white rounded-xl p-3 mb-3 shadow-lg">
                    <img src="assets/logo.png" alt="BoConcept" class="h-16 w-auto">
                </div>
                <h1 class="text-2xl font-bold">Crusardi</h1>
                <span class="text-sm text-gray-400 mt-1">Sistema de Gestión de Stock</span>
            </div>

            <!-- Navegación -->
            <nav class="space-y-2">
                <?php if ($_SESSION['user_role'] === 'viewer'): ?>
                    <a href="index.php?page=stock_calculado" 
                       class="sidebar-link flex items-center px-4 py-3 rounded-lg <?php echo $page === 'stock_calculado' ? 'bg-indigo-600 text-white shadow-lg' : 'text-gray-300 hover:bg-gray-700 hover:text-white'; ?>">
                        <i class="fas fa-chart-line w-5"></i>
                        <span class="ml-3 font-medium">Visor de Stock</span>
                    </a>
                <?php else: ?>
                    <!-- Vista principal -->
                    <a href="index.php?page=stock_calculado" 
                       class="sidebar-link flex items-center px-4 py-3 rounded-lg <?php echo $page === 'stock_calculado' ? 'bg-indigo-600 text-white shadow-lg' : 'text-gray-300 hover:bg-gray-700 hover:text-white'; ?>">
                        <i class="fas fa-chart-line w-5"></i>
                        <span class="ml-3 font-medium">Visor de Stock</span>
                    </a>
                    
                    <!-- Sección de Importación -->
                    <div x-data="{ open: <?php echo in_array($page, ['importar_stock', 'importar_combinaciones', 'importar_productos']) ? 'true' : 'false'; ?> }" class="mt-6">
                        <button @click="open = !open" 
                                class="w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white transition-all duration-200">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-file-import w-5"></i>
                                <span class="font-medium">Importación</span>
                            </div>
                            <i class="fas fa-chevron-down transition-transform duration-200 text-sm" :class="{'rotate-180': open}"></i>
                        </button>
                        <div x-show="open" 
                             x-transition:enter="transition ease-out duration-200"
                             x-transition:enter-start="opacity-0 transform -translate-y-2"
                             x-transition:enter-end="opacity-100 transform translate-y-0"
                             class="mt-2 ml-8 space-y-1">
                            <a href="index.php?page=importar_stock" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'importar_stock' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-boxes w-4 mr-2"></i>Stock Físico
                            </a>
                            <a href="index.php?page=importar_combinaciones" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'importar_combinaciones' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-box-open w-4 mr-2"></i>KITS / Recetas
                            </a>
                            <a href="index.php?page=importar_productos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'importar_productos' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-shopping-cart w-4 mr-2"></i>Productos
                            </a>
                        </div>
                    </div>
                    
                    <!-- Sección de Configuración -->
                    <div x-data="{ open: <?php echo in_array($page, ['gestionar_categorias', 'cargar_imagenes', 'descuentos', 'excepciones', 'gestionar_excepciones_kits']) ? 'true' : 'false'; ?> }" class="mt-4">
                        <button @click="open = !open" 
                                class="w-full flex items-center justify-between px-4 py-3 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white transition-all duration-200">
                            <div class="flex items-center gap-3">
                                <i class="fas fa-cog w-5"></i>
                                <span class="font-medium">Configuración</span>
                            </div>
                            <i class="fas fa-chevron-down transition-transform duration-200 text-sm" :class="{'rotate-180': open}"></i>
                        </button>
                        <div x-show="open" 
                             x-transition:enter="transition ease-out duration-200"
                             x-transition:enter-start="opacity-0 transform -translate-y-2"
                             x-transition:enter-end="opacity-100 transform translate-y-0"
                             class="mt-2 ml-8 space-y-1">
                            <a href="index.php?page=gestionar_categorias" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'gestionar_categorias' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-folder-tree w-4 mr-2"></i>Categorías
                            </a>
                            <a href="index.php?page=cargar_imagenes" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'cargar_imagenes' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-images w-4 mr-2"></i>Imágenes
                            </a>
                            <a href="index.php?page=descuentos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'descuentos' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-percent w-4 mr-2"></i>Descuentos
                            </a>
                            <a href="index.php?page=excepciones" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'excepciones' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-exclamation-triangle w-4 mr-2"></i>Excepciones
                            </a>
                            <a href="index.php?page=productos_imperfectos" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'productos_imperfectos' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-layer-group w-4 mr-2"></i>Productos Imperfectos
                            </a>
                            <a href="index.php?page=gestionar_excepciones_kits" 
                               class="sidebar-link block px-4 py-2 rounded-md text-sm <?php echo $page === 'gestionar_excepciones_kits' ? 'bg-indigo-600 text-white' : 'text-gray-400 hover:bg-gray-700 hover:text-white'; ?>">
                                <i class="fas fa-layer-group w-4 mr-2"></i>Excepciones Combinaciones
                            </a>
                        </div>
                    </div>
                    
                    <?php if ($_SESSION['user_role'] === 'admin'): ?>
                        <hr class="my-6 border-gray-700">
                        <a href="index.php?page=usuarios" 
                           class="sidebar-link flex items-center px-4 py-3 rounded-lg <?php echo $page === 'usuarios' ? 'bg-indigo-600 text-white shadow-lg' : 'text-gray-300 hover:bg-gray-700 hover:text-white'; ?>">
                            <i class="fas fa-users-cog w-5"></i>
                            <span class="ml-3 font-medium">Usuarios</span>
                        </a>
                    <?php endif; ?>
                <?php endif; ?>
            </nav>
            
            <!-- Información del usuario -->
            <div class="absolute bottom-0 left-0 right-0 p-5 bg-gray-900/50 backdrop-blur">
                <div class="flex items-center space-x-3">
                    <div class="bg-indigo-600 rounded-full p-3">
                        <i class="fas fa-user text-white"></i>
                    </div>
                    <div class="flex-1">
                        <p class="font-semibold text-white"><?= htmlspecialchars($_SESSION['user_name']); ?></p>
                        <p class="text-xs text-gray-400"><?= ucfirst($_SESSION['user_role']); ?></p>
                    </div>
                    <a href="auth/logout.php" 
                       class="text-red-400 hover:text-red-300 transition-colors"
                       title="Cerrar sesión">
                        <i class="fas fa-sign-out-alt"></i>
                    </a>
                </div>
            </div>
        </aside>

        <!-- Contenido principal -->
        <div class="flex-1 flex flex-col">
            <!-- Header móvil -->
            <header class="lg:hidden bg-white shadow-md p-4 flex justify-between items-center">
                <div class="flex items-center space-x-3">
                    <img src="assets/logo.png" alt="BoConcept" class="h-10 w-auto">
                    <span class="font-semibold text-gray-800">Crusardi</span>
                </div>
                <button @click="sidebarOpen = !sidebarOpen" 
                        class="text-gray-700 hover:text-gray-900 focus:outline-none p-2 rounded-lg hover:bg-gray-100">
                    <i class="fas fa-bars text-xl"></i>
                </button>
            </header>
            
            <!-- Contenido de la página -->
            <main class="flex-1 p-6 overflow-y-auto bg-gray-50">
                <div class="max-w-7xl mx-auto">
                    <?php if (file_exists($page_path)): ?>
                        <?php include $page_path; ?>
                    <?php else: ?>
                        <div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg shadow">
                            <div class="flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-exclamation-triangle text-red-400"></i>
                                </div>
                                <div class="ml-3">
                                    <p class="text-sm font-medium text-red-800">
                                        Error: La página solicitada no fue encontrada.
                                    </p>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>
</body>
</html>