<?php
// pages/importar_combinaciones.php

$error_message = null;
$success_message = null;
$current_role = $_SESSION['user_role'];

// Recuperar mensajes de sesión
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}

if (isset($_SESSION['import_error'])) {
    $error_message = $_SESSION['import_error'];
    unset($_SESSION['import_error']);
}

// Mostrar reporte de importación si existe
if (isset($_GET['report']) && isset($_SESSION['import_report'])) {
    $report = $_SESSION['import_report'];
    $success_message = "Importación completada: {$report['imported_count']} nuevos KITS importados";
    unset($_SESSION['import_report']);
}

// CÁLCULO DE INDICADORES (KPIs)
$kpi_total_relations = $pdo->query("SELECT COUNT(*) FROM lmat_relations")->fetchColumn();
$kpi_total_kits = $pdo->query("SELECT COUNT(DISTINCT lmat) FROM lmat_relations")->fetchColumn();
$kpi_avg_parts = ($kpi_total_kits > 0) ? round($kpi_total_relations / $kpi_total_kits, 2) : 0;

// Productos únicos en kits
$kpi_unique_products = $pdo->query("SELECT COUNT(DISTINCT parte) FROM lmat_relations")->fetchColumn();

// KPI de excepciones activas
$kpi_excepciones = $pdo->query("
    SELECT COUNT(DISTINCT kce.codigo_articulo) as total,
           COUNT(DISTINCT kce.kit_referencia) as kits_afectados
    FROM kit_component_exceptions kce
    WHERE kce.activo = 1
")->fetch();

// Componentes con stock que son excepciones
$kpi_stock_excepciones = $pdo->query("
    SELECT COUNT(DISTINCT sa.referencia) as total,
           SUM(sa.cantidad) as cantidad_total
    FROM stock_data sa
    JOIN kit_component_exceptions kce ON sa.referencia = kce.codigo_articulo
    WHERE kce.activo = 1 AND sa.cantidad > 0
")->fetch();

// Obtener las excepciones activas
$excepciones_query = $pdo->prepare("
    SELECT codigo_articulo, etiqueta_como, precio_individual 
    FROM kit_component_exceptions 
    WHERE activo = 1
");
$excepciones_query->execute();
$excepciones_activas = [];
while ($exc = $excepciones_query->fetch()) {
    $excepciones_activas[$exc['codigo_articulo']] = $exc;
}

// LÓGICA DE PAGINACIÓN Y BÚSQUEDA
$items_per_page = 50;
$current_page = isset($_GET['page_num']) ? (int)$_GET['page_num'] : 1;
if ($current_page < 1) $current_page = 1;
$search_term = isset($_GET['search']) ? trim($_GET['search']) : '';
$sort_by = isset($_GET['sort']) ? $_GET['sort'] : 'alpha_asc';

$relations_grouped = [];
$total_kits_filtered = 0;
$total_pages = 0;

try {
    $where_clauses = [];
    $params = [];
    
    if ($search_term) {
        $where_clauses[] = "(lr.lmat LIKE :search OR lr.parte LIKE :search2)";
        $params[':search'] = '%' . $search_term . '%';
        $params[':search2'] = '%' . $search_term . '%';
    }
    
    $base_where_sql = !empty($where_clauses) ? 'WHERE ' . implode(' AND ', $where_clauses) : '';
    
    // Contar total de kits filtrados
    $count_sql = "SELECT COUNT(DISTINCT lr.lmat) as total FROM lmat_relations lr " . $base_where_sql;
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total_kits_filtered = $count_stmt->fetchColumn();
    
    $total_pages = $total_kits_filtered > 0 ? ceil($total_kits_filtered / $items_per_page) : 1;
    if ($current_page > $total_pages && $total_pages > 0) $current_page = $total_pages;
    $offset = ($current_page - 1) * $items_per_page;
    
    $order_by_sql = "ORDER BY ";
    switch ($sort_by) {
        case 'date_desc':
            $order_by_sql .= "creation_date DESC";
            break;
        case 'date_asc':
            $order_by_sql .= "creation_date ASC";
            break;
        case 'parts_desc':
            $order_by_sql .= "parts_count DESC";
            break;
        case 'parts_asc':
            $order_by_sql .= "parts_count ASC";
            break;
        default:
            $order_by_sql .= "lr.lmat ASC";
    }

    if ($total_kits_filtered > 0) {
        // Obtener kits de la página actual
        $kits_sql = "
            SELECT 
                lr.lmat,
                MIN(lr.fecha_creacion) as creation_date,
                COUNT(DISTINCT lr.parte) as parts_count
            FROM lmat_relations lr
            " . str_replace('lr.', 'lr.', $base_where_sql) . " 
            GROUP BY lr.lmat
            " . str_replace('lr.', '', $order_by_sql) . " 
            LIMIT :limit OFFSET :offset";
            
        $kits_stmt = $pdo->prepare($kits_sql);
        foreach ($params as $key => $val) { $kits_stmt->bindValue($key, $val); }
        $kits_stmt->bindValue(':limit', $items_per_page, PDO::PARAM_INT);
        $kits_stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $kits_stmt->execute();
        
        $page_kits_data = $kits_stmt->fetchAll(PDO::FETCH_ASSOC);
        $page_kits = array_column($page_kits_data, 'lmat');

        if (!empty($page_kits)) {
            // Obtener las partes de cada kit
            $in_placeholders = implode(',', array_fill(0, count($page_kits), '?'));
            $parts_sql = "
                SELECT 
                    lr.lmat, 
                    lr.parte, 
                    lr.qty_required
                FROM lmat_relations lr
                WHERE lr.lmat IN ($in_placeholders) 
                ORDER BY lr.lmat, lr.parte";
            $parts_stmt = $pdo->prepare($parts_sql);
            $parts_stmt->execute($page_kits);
            while ($row = $parts_stmt->fetch()) {
                $relations_grouped[$row['lmat']][] = $row;
            }
        }
    }
} catch (PDOException $e) {
    $error_message = "No se pudieron cargar los kits: " . $e->getMessage();
}
?>

<div class="space-y-6">
    <!-- Header -->
    <div class="bg-gradient-to-r from-indigo-600 to-purple-700 rounded-xl p-6 text-white shadow-xl">
        <div class="flex items-center justify-between">
            <div class="flex items-center gap-4">
                <div class="bg-white/20 backdrop-blur p-4 rounded-lg">
                    <i class="fas fa-box-open fa-3x"></i>
                </div>
                <div>
                    <h1 class="text-3xl font-bold">Gestión de KITS</h1>
                    <p class="text-purple-100 mt-1">Administra las composiciones y recetas de productos</p>
                </div>
            </div>
            <div class="flex gap-3">
                <?php if (in_array($current_role, ['admin', 'editor'])): ?>
                <a href="index.php?page=gestionar_excepciones_kits" 
                   class="bg-yellow-500 text-white font-semibold px-6 py-3 rounded-lg hover:bg-yellow-600 transition transform hover:scale-105 shadow-lg">
                    <i class="fas fa-exclamation-triangle mr-2"></i>Excepciones
                </a>
                <button onclick="openModal('import-export-modal')" 
                        class="bg-white text-purple-700 font-semibold px-6 py-3 rounded-lg hover:bg-purple-50 transition transform hover:scale-105 shadow-lg">
                    <i class="fas fa-exchange-alt mr-2"></i>Importar / Exportar
                </button>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Mensajes -->
    <?php if ($error_message): ?>
        <div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg shadow-md animate-fade-in">
            <div class="flex items-center">
                <div class="bg-red-500 text-white rounded-full p-2 mr-3">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <div><?php echo $error_message; ?></div>
            </div>
        </div>
    <?php endif; ?>
    
    <?php if ($success_message): ?>
        <div class="bg-green-50 border-l-4 border-green-500 p-4 rounded-lg shadow-md animate-fade-in">
            <div class="flex items-center">
                <div class="bg-green-500 text-white rounded-full p-2 mr-3">
                    <i class="fas fa-check"></i>
                </div>
                <div>
                    <p class="font-bold text-green-800">Operación Exitosa</p>
                    <p class="text-green-700"><?php echo htmlspecialchars($success_message); ?></p>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <!-- KPIs -->
    <div class="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-6 gap-4">
        <div class="bg-white p-5 rounded-lg shadow-md flex items-center gap-4 transform transition hover:scale-105">
            <div class="bg-purple-100 text-purple-600 p-3 rounded-full">
                <i class="fas fa-box fa-lg"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Total de KITS</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo number_format($kpi_total_kits); ?></p>
            </div>
        </div>
        
        <div class="bg-white p-5 rounded-lg shadow-md flex items-center gap-4 transform transition hover:scale-105">
            <div class="bg-blue-100 text-blue-600 p-3 rounded-full">
                <i class="fas fa-sitemap fa-lg"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Total Composiciones</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo number_format($kpi_total_relations); ?></p>
            </div>
        </div>
        
        <div class="bg-white p-5 rounded-lg shadow-md flex items-center gap-4 transform transition hover:scale-105">
            <div class="bg-yellow-100 text-yellow-600 p-3 rounded-full">
                <i class="fas fa-cubes fa-lg"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Productos Únicos</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo number_format($kpi_unique_products); ?></p>
            </div>
        </div>
        
        <div class="bg-white p-5 rounded-lg shadow-md flex items-center gap-4 transform transition hover:scale-105">
            <div class="bg-green-100 text-green-600 p-3 rounded-full">
                <i class="fas fa-percentage fa-lg"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Promedio Partes/KIT</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo $kpi_avg_parts; ?></p>
            </div>
        </div>
        
        <div class="bg-white p-5 rounded-lg shadow-md flex items-center gap-4 transform transition hover:scale-105">
            <div class="bg-orange-100 text-orange-600 p-3 rounded-full">
                <i class="fas fa-exclamation-triangle fa-lg"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Excepciones Activas</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo number_format($kpi_excepciones['total']); ?></p>
                <p class="text-xs text-gray-400">En <?php echo $kpi_excepciones['kits_afectados']; ?> kits</p>
            </div>
        </div>

        <div class="bg-white p-5 rounded-lg shadow-md flex items-center gap-4 transform transition hover:scale-105">
            <div class="bg-teal-100 text-teal-600 p-3 rounded-full">
                <i class="fas fa-store fa-lg"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Partes en Stock</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo number_format($kpi_stock_excepciones['total']); ?></p>
                <p class="text-xs text-gray-400"><?php echo number_format($kpi_stock_excepciones['cantidad_total']); ?> unid.</p>
            </div>
        </div>
    </div>
    <!-- Filtros y búsqueda -->
    <div class="bg-white p-6 rounded-lg shadow-md">
        <form method="GET" action="index.php">
            <input type="hidden" name="page" value="importar_combinaciones">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div class="md:col-span-2 relative">
                    <label class="block text-xs font-medium text-gray-700 mb-1">Buscar</label>
                    <div class="relative">
                        <i class="fas fa-search absolute left-3 top-1/2 -translate-y-1/2 text-gray-400"></i>
                        <input type="text" name="search" 
                               placeholder="Buscar por código de KIT o componente..." 
                               class="w-full pl-10 p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500" 
                               value="<?php echo htmlspecialchars($search_term, ENT_QUOTES); ?>">
                    </div>
                </div>
                
                <div>
                    <label class="block text-xs font-medium text-gray-700 mb-1">Ordenar por</label>
                    <select name="sort" class="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500">
                        <option value="alpha_asc" <?php if($sort_by === 'alpha_asc') echo 'selected'; ?>>Código (A-Z)</option>
                        <option value="date_desc" <?php if($sort_by === 'date_desc') echo 'selected'; ?>>Más Recientes</option>
                        <option value="date_asc" <?php if($sort_by === 'date_asc') echo 'selected'; ?>>Más Antiguos</option>
                        <option value="parts_desc" <?php if($sort_by === 'parts_desc') echo 'selected'; ?>>Más Componentes</option>
                        <option value="parts_asc" <?php if($sort_by === 'parts_asc') echo 'selected'; ?>>Menos Componentes</option>
                    </select>
                </div>
            </div>
            
            <div class="flex justify-between items-center mt-4">
                <p class="text-sm text-gray-600">
                    Mostrando <?php echo number_format($total_kits_filtered); ?> KITS
                </p>
                <div class="flex gap-2">
                    <a href="index.php?page=importar_combinaciones" 
                       class="bg-gray-200 text-gray-700 font-semibold px-6 py-2 rounded-lg hover:bg-gray-300">
                        <i class="fas fa-times mr-2"></i>Limpiar
                    </a>
                    <button type="submit" class="bg-purple-600 text-white font-semibold px-6 py-2 rounded-lg hover:bg-purple-700">
                        <i class="fas fa-filter mr-2"></i>Aplicar Filtros
                    </button>
                </div>
            </div>
        </form>
        
        <!-- Selección y exportación -->
        <div class="flex justify-between items-center mt-4 pt-4 border-t">
            <div>
                <label class="flex items-center text-sm">
                    <input type="checkbox" id="select-all" class="h-4 w-4 rounded border-gray-300 mr-2">
                    Seleccionar todos (<?php echo count($relations_grouped); ?> en esta página)
                </label>
            </div>
            <button id="export-selected-btn" 
                    class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 font-semibold disabled:bg-gray-400 disabled:cursor-not-allowed" 
                    disabled>
                <i class="fas fa-download mr-2"></i>Exportar Selección (<span id="selected-count">0</span>)
            </button>
        </div>

        <!-- Lista de KITS -->
        <div id="relations-list" class="space-y-3 mt-6">
            <?php if (empty($relations_grouped)): ?>
                <div class="text-center py-12">
                    <i class="fas fa-box-open text-6xl text-gray-300 mb-4"></i>
                    <p class="text-gray-500">No se encontraron KITS que coincidan con la búsqueda.</p>
                </div>
            <?php else: ?>
                <?php foreach ($page_kits_data as $kit_data): ?>
                    <?php 
                    $kit = $kit_data['lmat'];
                    $parts = $relations_grouped[$kit] ?? [];
                    
                    // Contar excepciones en este kit
                    $excepciones_en_kit = 0;
                    foreach ($parts as $part) {
                        if (isset($excepciones_activas[$part['parte']])) {
                            $excepciones_en_kit++;
                        }
                    }
                    ?>
                    <details class="bg-white rounded-lg border border-gray-200 shadow-sm hover:shadow-md transition-shadow">
                        <summary class="p-4 cursor-pointer hover:bg-gray-50">
                            <div class="flex justify-between items-center">
                                <div class="flex items-center gap-4">
                                    <input type="checkbox" 
                                           class="kit-checkbox h-5 w-5 rounded border-gray-300" 
                                           value="<?php echo htmlspecialchars($kit); ?>" 
                                           onclick="event.stopPropagation()">
                                    
                                    <div>
                                        <div class="flex items-center gap-2">
                                            <span class="font-bold text-gray-800 text-lg"><?php echo htmlspecialchars($kit); ?></span>
                                            <?php if ($excepciones_en_kit > 0): ?>
                                                <span class="px-2 py-1 bg-yellow-100 text-yellow-700 rounded-full text-xs font-semibold">
                                                    <i class="fas fa-exclamation-triangle mr-1"></i><?php echo $excepciones_en_kit; ?> excepción(es)
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="text-xs text-gray-500 mt-1">
                                            Creado: <?php echo date('d/m/Y H:i', strtotime($kit_data['creation_date'])); ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="flex items-center gap-4">
                                    <span class="text-sm bg-purple-100 text-purple-700 font-semibold px-3 py-1 rounded-full">
                                        <?php echo $kit_data['parts_count']; ?> componente(s)
                                    </span>
                                    <i class="fas fa-chevron-right transition-transform details-arrow text-gray-400"></i>
                                </div>
                            </div>
                        </summary>
                        
                        <div class="border-t border-gray-200 p-4 bg-gray-50">
                            <table class="w-full text-sm">
                                <thead>
                                    <tr class="text-left border-b">
                                        <th class="p-2 font-medium text-gray-600">Código</th>
                                        <th class="p-2 font-medium text-gray-600 text-center">Cantidad</th>
                                        <th class="p-2 font-medium text-gray-600">Estado</th>
                                        <th class="p-2 font-medium text-gray-600 text-right">Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                <?php foreach ($parts as $rel): ?>
                                    <?php 
                                    $es_excepcion = isset($excepciones_activas[$rel['parte']]);
                                    $excepcion_data = $es_excepcion ? $excepciones_activas[$rel['parte']] : null;
                                    ?>
                                    <tr class="border-b hover:bg-white <?php echo $es_excepcion ? 'bg-yellow-50' : ''; ?>">
                                        <td class="p-2 font-mono text-sm">
                                            <?php echo htmlspecialchars($rel['parte']); ?>
                                            <?php if ($es_excepcion): ?>
                                                <div class="text-xs text-yellow-700 mt-1">
                                                    <i class="fas fa-tag mr-1"></i>Se vende como: 
                                                    <span class="font-semibold"><?php echo htmlspecialchars($excepcion_data['etiqueta_como']); ?></span>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td class="p-2 text-center font-semibold">
                                            <?php echo $rel['qty_required']; ?>
                                        </td>
                                        <td class="p-2">
                                            <?php if ($es_excepcion): ?>
                                                <span class="inline-flex items-center gap-1 px-2 py-1 bg-yellow-100 text-yellow-800 rounded-full text-xs font-medium">
                                                    <i class="fas fa-store"></i>
                                                    Venta individual activa
                                                </span>
                                                <?php if ($excepcion_data['precio_individual']): ?>
                                                    <div class="text-xs text-gray-600 mt-1">
                                                        Precio: $<?php echo number_format($excepcion_data['precio_individual'], 0, ',', '.'); ?>
                                                    </div>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <span class="text-xs text-gray-500">Solo en kit</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="p-2 text-right">
                                            <?php if (in_array($current_role, ['admin', 'editor'])): ?>
                                                <?php if ($es_excepcion): ?>
                                                    <a href="index.php?page=gestionar_excepciones_kits" 
                                                       class="text-blue-600 hover:text-blue-800 text-xs font-medium">
                                                        <i class="fas fa-edit mr-1"></i>Editar excepción
                                                    </a>
                                                <?php else: ?>
                                                    <button onclick="crearExcepcionRapida('<?php echo htmlspecialchars($rel['parte']); ?>', '<?php echo htmlspecialchars($kit); ?>')" 
                                                            class="text-green-600 hover:text-green-800 text-xs font-medium">
                                                        <i class="fas fa-plus-circle mr-1"></i>Crear excepción
                                                    </button>
                                                <?php endif; ?>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                </tbody>
                            </table>
                            
                            <?php if ($excepciones_en_kit > 0): ?>
                                <div class="mt-3 p-3 bg-yellow-100 border border-yellow-300 rounded-lg">
                                    <p class="text-sm text-yellow-800">
                                        <i class="fas fa-info-circle mr-2"></i>
                                        Este kit tiene <strong><?php echo $excepciones_en_kit; ?></strong> componente(s) que también se pueden vender individualmente.
                                    </p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </details>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
        
        <!-- Paginación -->
        <?php if ($total_pages > 1): ?>
        <div class="mt-8 flex flex-wrap justify-center items-center space-x-1">
            <?php
                $query_params = $_GET;
                unset($query_params['page_num']);
                $base_link = "?page=importar_combinaciones&" . http_build_query($query_params);
            ?>
            <a href="<?php echo $base_link; ?>&page_num=<?php echo $current_page - 1; ?>" 
               class="<?php echo $current_page <= 1 ? 'pointer-events-none text-gray-400' : 'text-gray-700 hover:bg-gray-200'; ?> px-3 py-2 bg-white border rounded-md">
                <i class="fas fa-chevron-left"></i>
            </a>
            
            <?php
            $max_links = 7;
            $start = max(1, $current_page - floor($max_links / 2));
            $end = min($total_pages, $start + $max_links - 1);
            if ($end - $start + 1 < $max_links) { 
                $start = max(1, $end - $max_links + 1); 
            }
            
            for ($i = $start; $i <= $end; $i++): ?>
                <a href="<?php echo $base_link; ?>&page_num=<?php echo $i; ?>" 
                   class="px-3 py-2 border rounded-md <?php echo ($i == $current_page) ? 'bg-purple-600 text-white border-purple-600' : 'bg-white text-gray-700 hover:bg-gray-200'; ?>">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>
            
            <a href="<?php echo $base_link; ?>&page_num=<?php echo $current_page + 1; ?>" 
               class="<?php echo $current_page >= $total_pages ? 'pointer-events-none text-gray-400' : 'text-gray-700 hover:bg-gray-200'; ?> px-3 py-2 bg-white border rounded-md">
                <i class="fas fa-chevron-right"></i>
            </a>
        </div>
        <?php endif; ?>
    </div>
</div>
<!-- Modal de Importación/Exportación -->
<div id="import-export-modal" class="fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center p-4 hidden z-50">
    <div class="bg-white rounded-xl shadow-2xl p-6 w-full max-w-4xl max-h-[90vh] overflow-hidden">
        <div class="flex justify-between items-center border-b pb-3 mb-4">
            <h3 class="text-xl font-bold text-gray-800">Importar / Exportar KITS</h3>
            <button onclick="closeModal('import-export-modal')" class="text-gray-500 hover:text-gray-800 text-2xl">&times;</button>
        </div>
        
        <div class="grid md:grid-cols-2 gap-6 overflow-y-auto max-h-[calc(90vh-200px)]">
            <div class="space-y-6">
                <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <h3 class="font-semibold text-lg text-blue-800">
                        <i class="fas fa-file-alt mr-2"></i>Importar desde TXT (Axapta)
                    </h3>
                    <p class="text-sm text-blue-700 mb-4">Formato jerárquico con indentación</p>
                    
                    <form class="import-form" method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="relaciones_import_ajax">
                        <input type="hidden" name="import_type" value="txt">
                        <label class="flex items-center justify-center w-full p-3 transition bg-white border-2 border-blue-300 border-dashed rounded-md cursor-pointer hover:border-blue-400">
                            <i class="fas fa-cloud-upload-alt mr-2 text-blue-600"></i>
                            <span class="text-sm text-blue-600 font-semibold">Seleccionar archivo .txt</span>
                            <input type="file" name="relations_file" class="hidden" accept=".txt" required>
                        </label>
                        <button type="submit" class="mt-3 w-full bg-blue-600 text-white px-4 py-3 rounded-lg hover:bg-blue-700 font-semibold flex items-center justify-center">
                            <i class="fas fa-upload mr-2"></i>Importar TXT
                        </button>
                    </form>
                </div>
                
                <div class="bg-cyan-50 border border-cyan-200 rounded-lg p-4">
                    <h3 class="font-semibold text-lg text-cyan-800">
                        <i class="fas fa-file-csv mr-2"></i>Importar desde CSV
                    </h3>
                    <p class="text-sm text-cyan-700 mb-4">Formato tabular simple</p>
                    
                    <div class="bg-white rounded p-3 mb-4">
                        <p class="text-xs font-semibold text-gray-700 mb-2">Columnas requeridas:</p>
                        <code class="text-xs bg-gray-100 p-2 rounded block">kit,parte,cantidad</code>
                    </div>
                    
                    <form class="import-form" method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="relaciones_import_ajax">
                        <input type="hidden" name="import_type" value="csv">
                        <label class="flex items-center justify-center w-full p-3 transition bg-white border-2 border-cyan-300 border-dashed rounded-md cursor-pointer hover:border-cyan-400">
                            <i class="fas fa-cloud-upload-alt mr-2 text-cyan-600"></i>
                            <span class="text-sm text-cyan-600 font-semibold">Seleccionar archivo .csv</span>
                            <input type="file" name="relations_file" class="hidden" accept=".csv" required>
                        </label>
                        <button type="submit" class="mt-3 w-full bg-cyan-600 text-white px-4 py-3 rounded-lg hover:bg-cyan-700 font-semibold flex items-center justify-center">
                            <i class="fas fa-file-csv mr-2"></i>Importar CSV
                        </button>
                    </form>
                </div>
            </div>
            
            <div>
                <div class="bg-green-50 border border-green-200 rounded-lg p-4 mb-4">
                    <h3 class="font-semibold text-lg text-green-800">
                        <i class="fas fa-download mr-2"></i>Exportar a CSV
                    </h3>
                    <p class="text-sm text-green-700 mb-4">Descarga todos los KITS con información completa</p>
                    <a href="index.php?page=importar_combinaciones&action=relaciones_export" 
                       class="block text-center w-full bg-green-600 text-white px-4 py-3 rounded-lg hover:bg-green-700 font-semibold">
                        <i class="fas fa-download mr-2"></i>Exportar Todos los KITS
                    </a>
                </div>

                <div id="terminal-container" class="hidden">
                    <h4 class="font-semibold text-gray-700 mb-2">
                        <i class="fas fa-terminal mr-2"></i>Progreso de la Importación:
                    </h4>
                    <pre id="terminal-output" class="bg-gray-900 text-green-400 text-xs p-4 rounded-lg h-64 overflow-y-auto font-mono"></pre>
                    <button id="terminal-close-btn" class="mt-3 w-full bg-gray-600 text-white py-2 rounded-lg hover:bg-gray-700 hidden">
                        <i class="fas fa-check mr-2"></i>Cerrar y Recargar
                    </button>
                </div>
                
                <div class="bg-amber-50 border border-amber-200 rounded-lg p-4 mt-4">
                    <h4 class="font-semibold text-amber-800 mb-2">
                        <i class="fas fa-info-circle mr-2"></i>Notas Importantes
                    </h4>
                    <ul class="text-sm text-amber-700 space-y-1 list-disc list-inside">
                        <li>La importación <strong>reemplazará</strong> todos los KITS existentes</li>
                        <li>Asegúrate de que los códigos coincidan con los productos existentes</li>
                        <li>Las excepciones configuradas se mantendrán después de la importación</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
details > summary { list-style: none; }
details > summary::-webkit-details-marker { display: none; }
details[open] .details-arrow { transform: rotate(90deg); }
details[open] > summary { background-color: #f9fafb; }

.animate-fade-in {
    animation: fadeIn 0.5s ease-in;
}
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}
</style>

<script>
function openModal(modalId) {
    document.getElementById('terminal-container').classList.add('hidden');
    document.getElementById('terminal-output').textContent = '';
    document.getElementById('terminal-close-btn').classList.add('hidden');
    document.querySelectorAll('.import-form button[type="submit"]').forEach(btn => btn.disabled = false);
    document.getElementById(modalId).classList.remove('hidden');
}

function closeModal(modalId) { 
    document.getElementById(modalId).classList.add('hidden'); 
}

function crearExcepcionRapida(codigoArticulo, kitReferencia) {
    if (confirm(`¿Desea crear una excepción para el componente ${codigoArticulo}?\n\nEsto permitirá venderlo individualmente cuando esté en stock.`)) {
        // Crear un formulario temporal para enviar
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = 'index.php?page=gestionar_excepciones_kits';
        
        // Agregar campos ocultos
        const campos = {
            'action': 'create_exception',
            'codigo_articulo': codigoArticulo,
            'nombre_articulo': `Componente de ${kitReferencia}`,
            'etiqueta_como': 'PARTE',
            'motivo_excepcion': 'Creado desde gestión de kits'
        };
        
        for (const [nombre, valor] of Object.entries(campos)) {
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = nombre;
            input.value = valor;
            form.appendChild(input);
        }
        
        document.body.appendChild(form);
        form.submit();
    }
}

// Manejo de importación con feedback visual
const importForms = document.querySelectorAll('.import-form');
importForms.forEach(form => {
    const fileInput = form.querySelector('input[type="file"]');
    const fileLabelSpan = form.querySelector('label > span');
    const originalLabelText = fileLabelSpan ? fileLabelSpan.textContent : '';

    if (fileInput && fileLabelSpan) {
        fileInput.addEventListener('change', function() {
            if (this.files.length > 0) {
                fileLabelSpan.innerHTML = '<i class="fas fa-file mr-2"></i>' + this.files[0].name;
            } else {
                fileLabelSpan.textContent = originalLabelText;
            }
        });
    }

    form.addEventListener('submit', function(event) {
        event.preventDefault();
        if (fileInput.files.length === 0) {
            alert('Por favor, selecciona un archivo para importar.');
            return;
        }
        
        const submitBtn = this.querySelector('button[type="submit"]');
        const terminalContainer = document.getElementById('terminal-container');
        const terminalOutput = document.getElementById('terminal-output');
        const terminalCloseBtn = document.getElementById('terminal-close-btn');
        
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Procesando...';
        
        const allButtons = document.querySelectorAll('.import-form button[type="submit"]');
        allButtons.forEach(btn => btn.disabled = true);

        terminalContainer.classList.remove('hidden');
        terminalOutput.textContent = 'Iniciando conexión con el servidor...\n';
        
        const formData = new FormData(this);
        const xhr = new XMLHttpRequest();
        
        xhr.open('POST', 'pages/actions/relaciones_import_ajax.php', true);
        
        let lastResponseLength = 0;
        xhr.onprogress = function() {
            const newResponse = xhr.responseText.substring(lastResponseLength);
            lastResponseLength = xhr.responseText.length;
            terminalOutput.textContent += newResponse;
            terminalOutput.scrollTop = terminalOutput.scrollHeight;
        };
        
        xhr.onload = function() {
            allButtons.forEach(btn => {
                btn.disabled = false;
                btn.innerHTML = btn.innerHTML.replace('<i class="fas fa-spinner fa-spin mr-2"></i>Procesando...', 
                                                    '<i class="fas fa-check mr-2"></i>Completado');
            });
            terminalCloseBtn.classList.remove('hidden');
        };
        
        xhr.send(formData);
    });
});

document.getElementById('terminal-close-btn').addEventListener('click', function() {
    window.location.href = 'index.php?page=importar_combinaciones&report=1';
});

// Manejo de selección múltiple
const selectAllCheckbox = document.getElementById('select-all');
const kitCheckboxes = document.querySelectorAll('.kit-checkbox');
const exportSelectedBtn = document.getElementById('export-selected-btn');
const selectedCountSpan = document.getElementById('selected-count');

function updateSelectionState() {
    const selected = document.querySelectorAll('.kit-checkbox:checked');
    selectedCountSpan.textContent = selected.length;
    exportSelectedBtn.disabled = selected.length === 0;
    if (kitCheckboxes.length > 0) {
        selectAllCheckbox.checked = selected.length === kitCheckboxes.length;
    } else {
        selectAllCheckbox.checked = false;
    }
}

selectAllCheckbox.addEventListener('change', (e) => {
    kitCheckboxes.forEach(cb => cb.checked = e.target.checked);
    updateSelectionState();
});

kitCheckboxes.forEach(cb => cb.addEventListener('change', updateSelectionState));

exportSelectedBtn.addEventListener('click', () => {
    const selected = Array.from(document.querySelectorAll('.kit-checkbox:checked')).map(cb => cb.value);
    if (selected.length > 0) {
        const params = new URLSearchParams({ page: 'importar_combinaciones', action: 'relaciones_export' });
        selected.forEach(kit => params.append('selected_lmats[]', kit));
        window.location.href = `index.php?${params.toString()}`;
    }
});

updateSelectionState();

// Cerrar modales al hacer clic fuera
document.addEventListener('click', function(event) {
    if (event.target.classList.contains('bg-opacity-50')) {
        event.target.classList.add('hidden');
    }
});
</script>