<?php
// pages/importar_productos.php

$error_message = null;
$success_message = null;
$current_role = $_SESSION['user_role'];
$log_messages = [];

// URL base para las imágenes
define('IMAGE_BASE_URL', 'https://imagenes.crusardi.net/boconcept/STOCKCOMERCIAL/');

// Verificar y crear columnas si no existen
try {
    // Verificar si existe categoria1
    $check_col1 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'categoria1'");
    if ($check_col1->rowCount() == 0) {
        $pdo->exec("ALTER TABLE references_data ADD COLUMN categoria1 VARCHAR(100)");
    }
    
    // Verificar si existe categoria2
    $check_col2 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'categoria2'");
    if ($check_col2->rowCount() == 0) {
        $pdo->exec("ALTER TABLE references_data ADD COLUMN categoria2 VARCHAR(100)");
    }
    
    // Verificar si existe tipo_articulo
    $check_col3 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'tipo_articulo'");
    if ($check_col3->rowCount() == 0) {
        $pdo->exec("ALTER TABLE references_data ADD COLUMN tipo_articulo VARCHAR(50)");
    }
    
    // Verificar si existe grupo_articulo
    $check_col4 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'grupo_articulo'");
    if ($check_col4->rowCount() == 0) {
        $pdo->exec("ALTER TABLE references_data ADD COLUMN grupo_articulo VARCHAR(100)");
    }
    
    // Verificar si existe estado
    $check_col5 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'estado'");
    if ($check_col5->rowCount() == 0) {
        $pdo->exec("ALTER TABLE references_data ADD COLUMN estado VARCHAR(50)");
    }
    
    // Si existe la columna categoria antigua y categoria1 está vacía, migrar datos
    $check_old = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'categoria'");
    if ($check_old->rowCount() > 0) {
        $pdo->exec("UPDATE references_data SET categoria1 = categoria WHERE categoria1 IS NULL OR categoria1 = ''");
    }
} catch (PDOException $e) {
    // Ignorar errores si las columnas ya existen
}

// Verificar si las funciones ya existen antes de declararlas
if (!function_exists('normalize_header')) {
    function normalize_header($str) {
        $str = strtolower(trim($str));
        $unwanted_array = ['á' => 'a', 'é' => 'e', 'í' => 'i', 'ó' => 'o', 'ú' => 'u', ' ' => '_', "\t" => '_', "\n" => '_', "\r" => '_'];
        $str = strtr($str, $unwanted_array);
        return preg_replace('/_+/', '_', $str);
    }
}

if (!function_exists('mapear_categoria_db')) {
    // Función para mapear categorías desde la base de datos
    function mapear_categoria_db($pdo, $grupo_articulo) {
        $grupo_upper = strtoupper(trim($grupo_articulo));
        
        // Buscar en la tabla de mapeo actualizada
        $stmt = $pdo->prepare("
            SELECT 
                COALESCE(c1.nombre, cm.categoria1) as cat1, 
                COALESCE(c2.nombre, cm.categoria2) as cat2
            FROM categoria_mapping cm
            LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
            LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
            WHERE cm.grupo_articulo = ? AND cm.activo = 1
        ");
        $stmt->execute([$grupo_upper]);
        $result = $stmt->fetch();
        
        if ($result) {
            return ['cat1' => $result['cat1'] ?: 'Otros', 'cat2' => $result['cat2'] ?: 'General'];
        }
        
        // Si no hay coincidencia exacta, buscar coincidencia parcial
        $stmt = $pdo->prepare("
            SELECT 
                COALESCE(c1.nombre, cm.categoria1) as cat1, 
                COALESCE(c2.nombre, cm.categoria2) as cat2
            FROM categoria_mapping cm
            LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
            LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
            WHERE ? LIKE CONCAT('%', cm.grupo_articulo, '%') AND cm.activo = 1 
            ORDER BY LENGTH(cm.grupo_articulo) DESC LIMIT 1
        ");
        $stmt->execute([$grupo_upper]);
        $result = $stmt->fetch();
        
        if ($result) {
            return ['cat1' => $result['cat1'] ?: 'Otros', 'cat2' => $result['cat2'] ?: 'General'];
        }
        
        // Default
        return ['cat1' => 'Otros', 'cat2' => 'General'];
    }
}

// --- LÓGICA DE PROCESAMIENTO DE ACCIONES (POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && in_array($current_role, ['admin', 'editor'])) {
    $action = $_POST['action'] ?? '';

    try {
        if ($action === 'import_paste' && !empty($_POST['paste_data'])) {
            $startTime = microtime(true);
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Proceso de importación iniciado.";
            
            // Procesar datos pegados
            $paste_data = $_POST['paste_data'];
            // Reemplazar cualquier combinación de retornos de carro por saltos de línea simples
            $paste_data = str_replace("\r\n", "\n", $paste_data);
            $paste_data = str_replace("\r", "\n", $paste_data);
            
            $lines = explode("\n", $paste_data);
            if (count($lines) < 2) {
                throw new Exception("No se detectaron datos válidos. Asegúrate de copiar con encabezados.");
            }
            
            // Detectar delimitador (siempre es tabulador en Axapta)
            $delimiter = "\t";
            
            // Parsear encabezados
            $headers = str_getcsv($lines[0], $delimiter);
            $normalized_headers = array_map('normalize_header', $headers);
            
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Detectadas " . count($headers) . " columnas";
            
            // Buscar índices de columnas necesarias
            $codigo_index = array_search('codigo_de_articulo', $normalized_headers);
            $nombre_index = array_search('nombre_del_articulo', $normalized_headers);
            $grupo_index = array_search('grupo_de_articulos', $normalized_headers);
            $tipo_index = array_search('tipo_de_articulo', $normalized_headers);
            $estado_index = array_search('estado', $normalized_headers);
            $precio_index = array_search('precio', $normalized_headers);
            
            // Si no encuentra "precio", buscar "precio_de_venta"
            if ($precio_index === false) {
                $precio_index = array_search('precio_de_venta', $normalized_headers);
            }
            
            if ($codigo_index === false) {
                throw new Exception("No se encontró la columna 'Código de artículo'");
            }
            
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Columnas mapeadas correctamente";
            
            $pdo->beginTransaction();
            $processed = 0;
            $created = 0;
            $updated = 0;
            $errors = 0;
            
            // Preparar statements
            $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM references_data WHERE referencia = ?");
            $insert_stmt = $pdo->prepare("
                INSERT INTO references_data (referencia, nombre, categoria1, categoria2, tipo_articulo, grupo_articulo, estado, url_imagen, precio, precio_final) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $update_stmt = $pdo->prepare("
                UPDATE references_data 
                SET nombre = ?, categoria1 = ?, categoria2 = ?, tipo_articulo = ?, grupo_articulo = ?, estado = ?, url_imagen = ?, precio = ?, precio_final = ? 
                WHERE referencia = ?
            ");
            
            // Procesar líneas
            for ($i = 1; $i < count($lines); $i++) {
                $line = trim($lines[$i]);
                if (empty($line)) continue;
                
                $data = str_getcsv($line, $delimiter);
                if (count($data) <= $codigo_index) continue;
                
                try {
                    $codigo = trim($data[$codigo_index] ?? '');
                    if (empty($codigo)) continue;
                    
                    $nombre = $nombre_index !== false ? trim($data[$nombre_index] ?? '') : '';
                    $grupo = $grupo_index !== false ? trim($data[$grupo_index] ?? '') : '';
                    $tipo = $tipo_index !== false ? trim($data[$tipo_index] ?? '') : '';
                    $estado = $estado_index !== false ? trim($data[$estado_index] ?? '') : '';
                    $precio_str = $precio_index !== false ? trim($data[$precio_index] ?? '0') : '0';
                    
                    // Limpiar precio (remover puntos de miles y cambiar coma por punto)
                    $precio_str = str_replace('.', '', $precio_str); // Remover puntos de miles
                    $precio_str = str_replace(',', '.', $precio_str); // Cambiar coma decimal por punto
                    $precio = (float)$precio_str;
                    
                    // Generar URL de imagen automáticamente
                    $url_imagen = IMAGE_BASE_URL . $codigo . '.jpg';
                    
                    // Mapear categorías usando la base de datos
                    $categorias = mapear_categoria_db($pdo, $grupo);
                    
                    // Verificar si existe
                    $check_stmt->execute([$codigo]);
                    $exists = $check_stmt->fetchColumn() > 0;
                    
                    if ($exists) {
                        $update_stmt->execute([
                            $nombre, 
                            $categorias['cat1'], 
                            $categorias['cat2'], 
                            $tipo, 
                            $grupo,
                            $estado,
                            $url_imagen,
                            $precio, 
                            $precio, 
                            $codigo
                        ]);
                        $updated++;
                        $log_messages[] = "[UPDATE] " . date('H:i:s') . " - Actualizado: $codigo (Estado: $estado)";
                    } else {
                        $insert_stmt->execute([
                            $codigo, 
                            $nombre, 
                            $categorias['cat1'], 
                            $categorias['cat2'], 
                            $tipo, 
                            $grupo,
                            $estado,
                            $url_imagen,
                            $precio, 
                            $precio
                        ]);
                        $created++;
                        $log_messages[] = "[CREATE] " . date('H:i:s') . " - Creado: $codigo (Estado: $estado)";
                    }
                    $processed++;
                    
                } catch (Exception $e) {
                    $errors++;
                    $log_messages[] = "[ERROR] " . date('H:i:s') . " - Error en línea $i: " . $e->getMessage();
                }
            }
            
            $pdo->commit();
            
            $endTime = microtime(true);
            $executionTime = round($endTime - $startTime, 2);
            
            $log_messages[] = "[OK] " . date('H:i:s') . " - Importación completada en {$executionTime}s";
            $log_messages[] = "[SUCCESS] " . date('H:i:s') . " - Procesados: $processed | Creados: $created | Actualizados: $updated | Errores: $errors";
            
            $_SESSION['success_message'] = "Importación exitosa: $processed productos procesados ($created nuevos, $updated actualizados)";
            $_SESSION['import_logs'] = $log_messages;
            
            @unlink(__DIR__ . '/../cache/public_catalog_data.json');
            
            header('Location: index.php?page=importar_productos');
            exit;
        }
        elseif ($action === 'create') {
            $precio = (float)($_POST['precio'] ?? 0);
            $referencia = trim($_POST['referencia']);
            $url_imagen = IMAGE_BASE_URL . $referencia . '.jpg';
            
            // Obtener categorías del select si se proporcionan
            $cat1_id = $_POST['categoria1_id'] ?? null;
            $cat2_id = $_POST['categoria2_id'] ?? null;
            
            $categoria1 = $_POST['categoria1'] ?? 'Otros';
            $categoria2 = $_POST['categoria2'] ?? 'General';
            
            if ($cat1_id) {
                $stmt = $pdo->prepare("SELECT nombre FROM categorias_nivel1 WHERE id = ?");
                $stmt->execute([$cat1_id]);
                $result = $stmt->fetch();
                if ($result) $categoria1 = $result['nombre'];
            }
            
            if ($cat2_id) {
                $stmt = $pdo->prepare("SELECT nombre FROM categorias_nivel2 WHERE id = ?");
                $stmt->execute([$cat2_id]);
                $result = $stmt->fetch();
                if ($result) $categoria2 = $result['nombre'];
            }
            
            $stmt = $pdo->prepare("
                INSERT INTO references_data (referencia, nombre, categoria1, categoria2, tipo_articulo, grupo_articulo, estado, url_imagen, precio, precio_final) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $referencia, 
                trim($_POST['nombre']), 
                $categoria1, 
                $categoria2,
                trim($_POST['tipo_articulo'] ?? 'Artículo'),
                trim($_POST['grupo_articulo'] ?? ''),
                trim($_POST['estado'] ?? ''),
                $url_imagen, 
                $precio, 
                $precio
            ]);
            $success_message = "Producto '" . htmlspecialchars($referencia) . "' creado con éxito.";
        }
        elseif ($action === 'update') {
            $precio = (float)($_POST['precio'] ?? 0);
            $referencia = trim($_POST['referencia']);
            
            // Si el checkbox de auto_url está marcado, regenerar la URL
            if (isset($_POST['auto_url'])) {
                $url_imagen = IMAGE_BASE_URL . $referencia . '.jpg';
            } else {
                $url_imagen = trim($_POST['url_imagen']);
            }
            
            $stmt = $pdo->prepare("
                UPDATE references_data 
                SET nombre = ?, categoria1 = ?, categoria2 = ?, tipo_articulo = ?, grupo_articulo = ?, estado = ?, url_imagen = ?, precio = ?, precio_final = ? 
                WHERE referencia = ?
            ");
            $stmt->execute([
                trim($_POST['nombre']), 
                trim($_POST['categoria1']), 
                trim($_POST['categoria2']),
                trim($_POST['tipo_articulo'] ?? 'Artículo'),
                trim($_POST['grupo_articulo'] ?? ''),
                trim($_POST['estado'] ?? ''),
                $url_imagen, 
                $precio, 
                $precio, 
                $referencia
            ]);
            $success_message = "Producto '" . htmlspecialchars($referencia) . "' actualizado con éxito.";
        }
        elseif ($action === 'delete' && $current_role === 'admin') {
            $stmt = $pdo->prepare("DELETE FROM references_data WHERE referencia = ?");
            $stmt->execute([trim($_POST['referencia'])]);
            $success_message = "Producto '" . htmlspecialchars($_POST['referencia']) . "' eliminado con éxito.";
        }

        if (isset($success_message)) {
            $_SESSION['success_message'] = $success_message;
            @unlink(__DIR__ . '/../cache/public_catalog_data.json');
            header('Location: index.php?page=importar_productos');
            exit;
        }
    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
        $error_message = "<strong>Ocurrió un error:</strong> " . $e->getMessage();
        $log_messages[] = "[ERROR] " . date('H:i:s') . " - " . $e->getMessage();
    }
}

// Recuperar mensajes de sesión
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}

if (isset($_SESSION['import_logs'])) {
    $log_messages = $_SESSION['import_logs'];
    unset($_SESSION['import_logs']);
}

// --- LÓGICA DE BÚSQUEDA Y PAGINACIÓN ---
$products = [];
$total_items = 0;
$items_per_page = 50;
$search_term = $_GET['search'] ?? '';
$filter_grupo = $_GET['grupo'] ?? '';
$filter_cat1 = $_GET['cat1'] ?? '';
$filter_cat2 = $_GET['cat2'] ?? '';
$filter_estado = $_GET['estado'] ?? '';
$filter_precio_min = $_GET['precio_min'] ?? '';
$filter_precio_max = $_GET['precio_max'] ?? '';
$filter_imagen = $_GET['imagen'] ?? '';

try {
    $where_conditions = [];
    $params = [];
    
    if ($search_term) {
        $where_conditions[] = "(referencia LIKE ? OR nombre LIKE ?)";
        $search_like = '%' . $search_term . '%';
        $params = array_merge($params, [$search_like, $search_like]);
    }
    
    if ($filter_grupo) {
        $where_conditions[] = "grupo_articulo = ?";
        $params[] = $filter_grupo;
    }
    
    if ($filter_cat1) {
        $where_conditions[] = "categoria1 = ?";
        $params[] = $filter_cat1;
    }
    
    if ($filter_cat2) {
        $where_conditions[] = "categoria2 = ?";
        $params[] = $filter_cat2;
    }
    
    if ($filter_estado) {
        $where_conditions[] = "estado = ?";
        $params[] = $filter_estado;
    }
    
    if ($filter_precio_min !== '') {
        $where_conditions[] = "precio >= ?";
        $params[] = $filter_precio_min;
    }
    
    if ($filter_precio_max !== '') {
        $where_conditions[] = "precio <= ?";
        $params[] = $filter_precio_max;
    }
    
    if ($filter_imagen === 'con') {
        $where_conditions[] = "url_imagen IS NOT NULL AND url_imagen != ''";
    } elseif ($filter_imagen === 'sin') {
        $where_conditions[] = "(url_imagen IS NULL OR url_imagen = '')";
    }
    
    $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
    
    // Contar total
    $count_sql = "SELECT COUNT(*) FROM references_data " . $where_clause;
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total_items = $count_stmt->fetchColumn();
    
    // Paginación
    $total_pages = $total_items > 0 ? ceil($total_items / $items_per_page) : 1;
    $current_page_num = isset($_GET['page_num']) ? (int)$_GET['page_num'] : 1;
    if ($current_page_num > $total_pages && $total_pages > 0) $current_page_num = $total_pages;
    if ($current_page_num < 1) $current_page_num = 1;
    $offset = ($current_page_num - 1) * $items_per_page;
    
    // Obtener productos
    $fetch_sql = "SELECT referencia, nombre, categoria1, categoria2, tipo_articulo, grupo_articulo, estado, url_imagen, precio, precio_final 
                  FROM references_data " . $where_clause . " ORDER BY referencia LIMIT ? OFFSET ?";
    $products_stmt = $pdo->prepare($fetch_sql);
    $fetch_params = array_merge($params, [$items_per_page, $offset]);
    $products_stmt->execute($fetch_params);
    $products = $products_stmt->fetchAll();
    
    // Obtener valores únicos para filtros - CORRECTAMENTE de la base de datos
    // Grupos de artículos
    $grupos_stmt = $pdo->query("SELECT DISTINCT grupo_articulo FROM references_data WHERE grupo_articulo IS NOT NULL AND grupo_articulo != '' ORDER BY grupo_articulo");
    $grupos = $grupos_stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Categorías nivel 1 - desde la tabla correcta
    try {
        $cat1_stmt = $pdo->query("SELECT id, nombre FROM categorias_nivel1 WHERE activo = 1 ORDER BY orden, nombre");
        $categorias1 = $cat1_stmt->fetchAll();
    } catch (PDOException $e) {
        // Si la tabla no existe, usar valores únicos de references_data
        $cat1_stmt = $pdo->query("SELECT DISTINCT categoria1 FROM references_data WHERE categoria1 IS NOT NULL AND categoria1 != '' ORDER BY categoria1");
        $categorias1 = array_map(function($cat) {
            return ['id' => null, 'nombre' => $cat];
        }, $cat1_stmt->fetchAll(PDO::FETCH_COLUMN));
    }
    
    // Categorías nivel 2 - desde la tabla correcta
    try {
        $cat2_stmt = $pdo->query("SELECT DISTINCT c2.id, c2.nombre, c1.nombre as cat1_nombre 
                                  FROM categorias_nivel2 c2 
                                  JOIN categorias_nivel1 c1 ON c2.categoria1_id = c1.id 
                                  WHERE c2.activo = 1 
                                  ORDER BY c1.orden, c1.nombre, c2.orden, c2.nombre");
        $categorias2_full = $cat2_stmt->fetchAll();
    } catch (PDOException $e) {
        // Si la tabla no existe, usar valores únicos de references_data
        $cat2_stmt = $pdo->query("SELECT DISTINCT categoria2 FROM references_data WHERE categoria2 IS NOT NULL AND categoria2 != '' ORDER BY categoria2");
        $categorias2_full = array_map(function($cat) {
            return ['id' => null, 'nombre' => $cat, 'cat1_nombre' => ''];
        }, $cat2_stmt->fetchAll(PDO::FETCH_COLUMN));
    }
    
    // Estados únicos
    $estados_stmt = $pdo->query("SELECT DISTINCT estado FROM references_data WHERE estado IS NOT NULL AND estado != '' ORDER BY estado");
    $estados = $estados_stmt->fetchAll(PDO::FETCH_COLUMN);
    
} catch (PDOException $e) {
    $error_message = "<strong>Error al cargar la lista de productos:</strong> " . $e->getMessage();
}
?>
<div class="space-y-6">
    <!-- Header mejorado -->
    <div class="bg-gradient-to-r from-purple-600 to-indigo-700 rounded-xl p-6 text-white shadow-xl">
        <div class="flex items-center justify-between">
            <div class="flex items-center gap-4">
                <div class="bg-white/20 backdrop-blur p-4 rounded-lg">
                    <i class="fas fa-book fa-3x"></i>
                </div>
                <div>
                    <h1 class="text-3xl font-bold">Gestión de Productos</h1>
                    <p class="text-purple-100 mt-1">Administra el catálogo de referencias con categorías y precios</p>
                </div>
            </div>
            <?php if (in_array($current_role, ['admin', 'editor'])): ?>
            <div class="flex items-center gap-2">
                <a href="index.php?page=gestionar_categorias" class="bg-white/20 backdrop-blur text-white font-semibold px-4 py-2 rounded-lg hover:bg-white/30 transition">
                    <i class="fas fa-tags mr-2"></i>Categorías
                </a>
                <button onclick="openModal('create-modal')" class="bg-white/20 backdrop-blur text-white font-semibold px-4 py-2 rounded-lg hover:bg-white/30 transition">
                    <i class="fas fa-plus mr-2"></i>Añadir
                </button>
                <button onclick="openModal('import-modal')" class="bg-white text-purple-700 font-semibold px-4 py-2 rounded-lg hover:bg-purple-50 transition">
                    <i class="fas fa-paste mr-2"></i>Importar desde Axapta
                </button>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Mensajes -->
    <?php if ($error_message): ?>
        <div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg shadow-md animate-fade-in">
            <div class="flex items-center">
                <div class="bg-red-500 text-white rounded-full p-2 mr-3">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <div><?php echo $error_message; ?></div>
            </div>
        </div>
    <?php endif; ?>
    
    <?php if ($success_message): ?>
        <div class="bg-green-50 border-l-4 border-green-500 p-4 rounded-lg shadow-md animate-fade-in">
            <div class="flex items-center">
                <div class="bg-green-500 text-white rounded-full p-2 mr-3">
                    <i class="fas fa-check"></i>
                </div>
                <div>
                    <p class="font-bold text-green-800">Operación Exitosa</p>
                    <p class="text-green-700"><?php echo htmlspecialchars($success_message); ?></p>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <!-- Consola de logs -->
    <?php if (!empty($log_messages)): ?>
    <div class="bg-gray-900 rounded-xl shadow-lg overflow-hidden animate-slide-up">
        <div class="bg-gray-800 px-6 py-3 flex items-center justify-between">
            <h3 class="text-white font-semibold flex items-center">
                <i class="fas fa-terminal mr-2"></i>Log de Importación
            </h3>
            <button onclick="this.parentElement.parentElement.style.display='none'" class="text-gray-400 hover:text-white">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="p-4 max-h-48 overflow-y-auto font-mono text-sm">
            <?php foreach ($log_messages as $msg): ?>
                <p class="mb-1 <?php
                    if (strpos($msg, '[OK]') !== false) echo 'text-green-400';
                    elseif (strpos($msg, '[SUCCESS]') !== false) echo 'text-green-300 font-bold';
                    elseif (strpos($msg, '[CREATE]') !== false) echo 'text-blue-400';
                    elseif (strpos($msg, '[UPDATE]') !== false) echo 'text-yellow-400';
                    elseif (strpos($msg, '[WARN]') !== false) echo 'text-yellow-400';
                    elseif (strpos($msg, '[ERROR]') !== false) echo 'text-red-400';
                    else echo 'text-gray-300';
                ?>"><?php echo htmlspecialchars($msg); ?></p>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- Filtros de búsqueda MEJORADOS -->
    <div class="bg-white p-4 rounded-lg shadow-md">
        <form method="GET" action="index.php" id="filter-form">
            <input type="hidden" name="page" value="importar_productos">
            
            <!-- Primera fila de filtros -->
            <div class="grid grid-cols-1 md:grid-cols-6 gap-4 mb-4">
                <div class="md:col-span-2 relative">
                    <label class="block text-xs font-medium text-gray-700 mb-1">Buscar</label>
                    <div class="relative">
                        <i class="fas fa-search absolute left-3 top-1/2 -translate-y-1/2 text-gray-400"></i>
                        <input type="text" name="search" placeholder="Referencia o nombre..." 
                               class="w-full pl-10 p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500" 
                               value="<?php echo htmlspecialchars($search_term); ?>">
                    </div>
                </div>
                
                <div>
                    <label class="block text-xs font-medium text-gray-700 mb-1">Grupo Artículo</label>
                    <select name="grupo" class="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500">
                        <option value="">Todos los grupos</option>
                        <?php foreach($grupos as $grupo): ?>
                            <option value="<?php echo htmlspecialchars($grupo); ?>" <?php echo $filter_grupo === $grupo ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($grupo); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div>
                    <label class="block text-xs font-medium text-gray-700 mb-1">Categoría 1</label>
                    <select name="cat1" id="filter-cat1" class="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500">
                        <option value="">Todas las categorías</option>
                        <?php foreach($categorias1 as $cat): ?>
                            <option value="<?php echo htmlspecialchars($cat['nombre']); ?>" <?php echo $filter_cat1 === $cat['nombre'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cat['nombre']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div>
                    <label class="block text-xs font-medium text-gray-700 mb-1">Categoría 2</label>
                    <select name="cat2" id="filter-cat2" class="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500">
                        <option value="">Todas las subcategorías</option>
                        <?php foreach($categorias2_full as $cat): ?>
                            <option value="<?php echo htmlspecialchars($cat['nombre']); ?>" 
                                    data-cat1="<?php echo htmlspecialchars($cat['cat1_nombre']); ?>"
                                    <?php echo $filter_cat2 === $cat['nombre'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cat['nombre']); ?>
                                <?php if ($cat['cat1_nombre']): ?>
                                    (<?php echo htmlspecialchars($cat['cat1_nombre']); ?>)
                                <?php endif; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div>
                    <label class="block text-xs font-medium text-gray-700 mb-1">Estado</label>
                    <select name="estado" class="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500">
                        <option value="">Todos los estados</option>
                        <option value="99" <?php echo $filter_estado === '99' ? 'selected' : ''; ?>>99 - Descontinuado</option>
                        <option value="activo" <?php echo $filter_estado === 'activo' ? 'selected' : ''; ?>>Activos (< 99)</option>
                        <?php foreach($estados as $estado): ?>
                            <?php if ($estado !== '99'): ?>
                                <option value="<?php echo htmlspecialchars($estado); ?>" <?php echo $filter_estado === $estado ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($estado); ?>
                                </option>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            
            <!-- Segunda fila de filtros -->
            <div class="grid grid-cols-1 md:grid-cols-6 gap-4">
                <div>
                    <label class="block text-xs font-medium text-gray-700 mb-1">Precio Min</label>
                    <input type="number" name="precio_min" step="0.01" placeholder="0.00"
                           class="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500" 
                           value="<?php echo htmlspecialchars($filter_precio_min); ?>">
                </div>
                
                <div>
                    <label class="block text-xs font-medium text-gray-700 mb-1">Precio Max</label>
                    <input type="number" name="precio_max" step="0.01" placeholder="999999.99"
                           class="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500" 
                           value="<?php echo htmlspecialchars($filter_precio_max); ?>">
                </div>
                
                <div>
                    <label class="block text-xs font-medium text-gray-700 mb-1">Imágenes</label>
                    <select name="imagen" class="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500">
                        <option value="">Todas</option>
                        <option value="con" <?php echo $filter_imagen === 'con' ? 'selected' : ''; ?>>Con imagen</option>
                        <option value="sin" <?php echo $filter_imagen === 'sin' ? 'selected' : ''; ?>>Sin imagen</option>
                    </select>
                </div>
                
                <div class="md:col-span-3 flex items-end gap-2">
                    <button type="submit" class="flex-1 bg-purple-600 text-white font-semibold px-6 py-2 rounded-lg hover:bg-purple-700">
                        <i class="fas fa-filter mr-2"></i>Aplicar Filtros
                    </button>
                    <a href="index.php?page=importar_productos" class="bg-gray-200 text-gray-700 font-semibold px-6 py-2 rounded-lg hover:bg-gray-300">
                        <i class="fas fa-times mr-2"></i>Limpiar
                    </a>
                </div>
            </div>
        </form>
    </div>

    <!-- Información de filtros aplicados -->
    <?php if ($search_term || $filter_grupo || $filter_cat1 || $filter_cat2 || $filter_estado || $filter_precio_min || $filter_precio_max || $filter_imagen): ?>
    <div class="bg-blue-50 border border-blue-200 rounded-lg p-3">
        <p class="text-sm text-blue-800">
            <i class="fas fa-info-circle mr-2"></i>
            Mostrando <?php echo number_format($total_items); ?> productos filtrados
            <?php if ($search_term): ?>
                • Búsqueda: "<strong><?php echo htmlspecialchars($search_term); ?></strong>"
            <?php endif; ?>
            <?php if ($filter_grupo): ?>
                • Grupo: <strong><?php echo htmlspecialchars($filter_grupo); ?></strong>
            <?php endif; ?>
            <?php if ($filter_cat1): ?>
                • Cat1: <strong><?php echo htmlspecialchars($filter_cat1); ?></strong>
            <?php endif; ?>
            <?php if ($filter_cat2): ?>
                • Cat2: <strong><?php echo htmlspecialchars($filter_cat2); ?></strong>
            <?php endif; ?>
            <?php if ($filter_estado): ?>
                • Estado: <strong><?php echo htmlspecialchars($filter_estado); ?></strong>
            <?php endif; ?>
            <?php if ($filter_precio_min || $filter_precio_max): ?>
                • Precio: <strong><?php echo $filter_precio_min ?: '0'; ?> - <?php echo $filter_precio_max ?: '∞'; ?></strong>
            <?php endif; ?>
            <?php if ($filter_imagen): ?>
                • <strong><?php echo $filter_imagen === 'con' ? 'Con imagen' : 'Sin imagen'; ?></strong>
            <?php endif; ?>
        </p>
    </div>
    <?php endif; ?>

    <!-- Tabla de productos -->
    <div class="bg-white rounded-lg shadow-md overflow-hidden">
        <div class="p-4 bg-gray-50 border-b">
            <h2 class="text-lg font-semibold text-gray-800">
                Productos (<?php echo number_format($total_items); ?> registros)
            </h2>
        </div>
        <div class="overflow-x-auto">
            <table class="w-full text-sm">
                <thead class="bg-gray-100">
                    <tr>
                        <th class="p-3 text-left font-semibold text-gray-600">Imagen</th>
                        <th class="p-3 text-left font-semibold text-gray-600">Referencia</th>
                        <th class="p-3 text-left font-semibold text-gray-600">Nombre</th>
                        <th class="p-3 text-left font-semibold text-gray-600">Grupo</th>
                        <th class="p-3 text-left font-semibold text-gray-600">Cat. 1</th>
                        <th class="p-3 text-left font-semibold text-gray-600">Cat. 2</th>
                        <th class="p-3 text-left font-semibold text-gray-600">Tipo</th>
                        <th class="p-3 text-left font-semibold text-gray-600">Estado</th>
                        <th class="p-3 text-left font-semibold text-gray-600">Precio</th>
                        <th class="p-3 text-right font-semibold text-gray-600">Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($products)): ?>
                        <tr><td colspan="10" class="text-center p-8 text-gray-500">No se encontraron productos.</td></tr>
                    <?php endif; ?>
                    <?php foreach ($products as $product): ?>
                    <tr class="border-b hover:bg-gray-50">
                        <form method="POST" action="index.php?page=importar_productos">
                            <input type="hidden" name="action" value="update">
                            <input type="hidden" name="referencia" value="<?php echo htmlspecialchars($product['referencia']); ?>">
                            <td class="p-2 text-center">
                                <?php 
                                $imagen_url = $product['url_imagen'] ?: IMAGE_BASE_URL . $product['referencia'] . '.jpg';
                                ?>
                                <div class="relative group">
                                    <img src="<?php echo htmlspecialchars($imagen_url); ?>" 
                                         alt="<?php echo htmlspecialchars($product['referencia']); ?>" 
                                         class="h-16 w-16 object-cover rounded-lg inline-block shadow-sm cursor-pointer"
                                         onerror="this.src='data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'64\' height=\'64\' viewBox=\'0 0 24 24\'%3E%3Cpath fill=\'%23ccc\' d=\'M21 19V5c0-1.1-.9-2-2-2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2zM8.5 13.5l2.5 3.01L14.5 12l4.5 6H5l3.5-4.5z\'/%3E%3C/svg%3E'"
                                         onclick="window.open('<?php echo htmlspecialchars($imagen_url); ?>', '_blank')">
                                    
                                    <!-- Tooltip con URL -->
                                    <div class="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 px-2 py-1 bg-gray-800 text-white text-xs rounded opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-10">
                                        <?php echo htmlspecialchars($product['referencia']); ?>.jpg
                                    </div>
                                </div>
                                
                                <?php if ($current_role !== 'viewer'): ?>
                                    <div class="mt-1">
                                        <label class="flex items-center justify-center text-xs">
                                            <input type="checkbox" name="auto_url" class="mr-1" checked>
                                            <span class="text-gray-600">Auto</span>
                                        </label>
                                    </div>
                                    <input type="hidden" name="url_imagen" value="<?php echo htmlspecialchars($imagen_url); ?>">
                                <?php endif; ?>
                            </td>
                            <td class="p-2">
                                <span class="font-mono font-semibold text-purple-600">
                                    <?php echo htmlspecialchars($product['referencia']); ?>
                                </span>
                            </td>
                            <td class="p-2">
                                <input type="text" name="nombre" 
                                       class="w-full p-1 border border-gray-200 rounded text-sm" 
                                       value="<?php echo htmlspecialchars($product['nombre'] ?? ''); ?>" 
                                       <?php echo ($current_role === 'viewer') ? 'readonly' : ''; ?>>
                            </td>
                            <td class="p-2">
                                <input type="text" name="grupo_articulo" 
                                       class="w-full p-1 border border-gray-200 rounded text-xs font-mono" 
                                       value="<?php echo htmlspecialchars($product['grupo_articulo'] ?? ''); ?>" 
                                       <?php echo ($current_role === 'viewer') ? 'readonly' : ''; ?>>
                            </td>
                            <td class="p-2">
                                <input type="text" name="categoria1" 
                                       class="w-full p-1 border border-gray-200 rounded text-sm" 
                                       value="<?php echo htmlspecialchars($product['categoria1'] ?? ''); ?>" 
                                       <?php echo ($current_role === 'viewer') ? 'readonly' : ''; ?>>
                            </td>
                            <td class="p-2">
                                <input type="text" name="categoria2" 
                                       class="w-full p-1 border border-gray-200 rounded text-sm" 
                                       value="<?php echo htmlspecialchars($product['categoria2'] ?? ''); ?>" 
                                       <?php echo ($current_role === 'viewer') ? 'readonly' : ''; ?>>
                            </td>
                            <td class="p-2">
                                <select name="tipo_articulo" 
                                        class="w-full p-1 border border-gray-200 rounded text-xs" 
                                        <?php echo ($current_role === 'viewer') ? 'disabled' : ''; ?>>
                                    <option value="Artículo" <?php echo ($product['tipo_articulo'] ?? '') === 'Artículo' ? 'selected' : ''; ?>>Artículo</option>
                                    <option value="L.MAT" <?php echo ($product['tipo_articulo'] ?? '') === 'L.MAT' ? 'selected' : ''; ?>>L.MAT</option>
                                </select>
                            </td>
                            <td class="p-2">
                                <?php 
                                $estado = $product['estado'] ?? '';
                                $estado_class = '';
                                if ($estado == '99') {
                                    $estado_class = 'bg-red-100 text-red-700';
                                } elseif (is_numeric($estado) && $estado < 99) {
                                    $estado_class = 'bg-green-100 text-green-700';
                                } else {
                                    $estado_class = 'bg-gray-100 text-gray-700';
                                }
                                ?>
                                <input type="text" name="estado" 
                                       class="w-full p-1 border border-gray-200 rounded text-xs text-center <?php echo $estado_class; ?>" 
                                       value="<?php echo htmlspecialchars($estado); ?>" 
                                       <?php echo ($current_role === 'viewer') ? 'readonly' : ''; ?>>
                            </td>
                            <td class="p-2">
                                <input type="number" step="0.01" name="precio" 
                                       class="w-full p-1 border border-gray-200 rounded text-sm font-semibold" 
                                       value="<?php echo htmlspecialchars($product['precio'] ?? '0'); ?>" 
                                       <?php echo ($current_role === 'viewer') ? 'readonly' : ''; ?>>
                            </td>
                            <td class="p-2 text-right">
                                <div class="flex justify-end space-x-1">
                                    <?php if (in_array($current_role, ['admin', 'editor'])): ?>
                                        <button type="submit" class="bg-blue-500 text-white px-2 py-1 rounded hover:bg-blue-600 transition text-xs" title="Guardar">
                                            <i class="fas fa-save"></i>
                                        </button>
                                    <?php endif; ?>
                                    <?php if ($current_role === 'admin'): ?>
                                        <button type="button" onclick="openDeleteModal('<?php echo htmlspecialchars($product['referencia']); ?>')" 
                                                class="bg-red-500 text-white px-2 py-1 rounded hover:bg-red-600 transition text-xs" title="Eliminar">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </form>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Paginación -->
    <div class="mt-6 flex justify-center items-center space-x-1">
        <?php if ($total_pages > 1): ?>
            <?php 
            $base_link = "?page=importar_productos" . 
                        "&search=" . urlencode($search_term) . 
                        "&grupo=" . urlencode($filter_grupo) .
                        "&cat1=" . urlencode($filter_cat1) . 
                        "&cat2=" . urlencode($filter_cat2) . 
                        "&estado=" . urlencode($filter_estado) .
                        "&precio_min=" . urlencode($filter_precio_min) .
                        "&precio_max=" . urlencode($filter_precio_max) .
                        "&imagen=" . urlencode($filter_imagen); 
            ?>
            <a href="<?php echo $base_link; ?>&page_num=<?php echo $current_page_num - 1; ?>" 
               class="<?php echo $current_page_num <= 1 ? 'pointer-events-none text-gray-400' : 'text-gray-700 hover:bg-gray-200'; ?> px-3 py-2 bg-white border rounded-md">
                <i class="fas fa-chevron-left"></i>
            </a>
            
            <?php
            $max_links = 7;
            $start = max(1, $current_page_num - floor($max_links / 2));
            $end = min($total_pages, $start + $max_links - 1);
            if ($end - $start + 1 < $max_links) {
                $start = max(1, $end - $max_links + 1);
            }
            
            for ($i = $start; $i <= $end; $i++): 
            ?>
                <a href="<?php echo $base_link; ?>&page_num=<?php echo $i; ?>" 
                   class="px-3 py-2 border rounded-md <?php echo $i == $current_page_num ? 'bg-purple-600 text-white border-purple-600' : 'bg-white text-gray-700 hover:bg-gray-200'; ?>">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>
            
            <a href="<?php echo $base_link; ?>&page_num=<?php echo $current_page_num + 1; ?>" 
               class="<?php echo $current_page_num >= $total_pages ? 'pointer-events-none text-gray-400' : 'text-gray-700 hover:bg-gray-200'; ?> px-3 py-2 bg-white border rounded-md">
                <i class="fas fa-chevron-right"></i>
            </a>
        <?php endif; ?>
    </div>
</div>

<!-- Modal de Importación desde Axapta -->
<div id="import-modal" class="fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center p-4 hidden z-50">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-4xl max-h-[90vh] overflow-hidden">
        <div class="bg-gradient-to-r from-purple-600 to-indigo-700 text-white p-6">
            <div class="flex justify-between items-center">
                <div>
                    <h2 class="text-2xl font-bold">Importar desde Axapta</h2>
                    <p class="text-purple-100 mt-1">Copia y pega los datos directamente desde el sistema</p>
                </div>
                <button onclick="closeModal('import-modal')" class="text-white/80 hover:text-white text-2xl">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        </div>
        
        <form method="POST" action="index.php?page=importar_productos" class="p-6 overflow-y-auto max-h-[calc(90vh-120px)]">
            <input type="hidden" name="action" value="import_paste">
            
            <div class="mb-4">
                <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">
                    <h4 class="font-semibold text-blue-800 mb-2">
                        <i class="fas fa-info-circle mr-2"></i>Instrucciones
                    </h4>
                    <ol class="text-sm text-blue-700 space-y-1 list-decimal list-inside">
                        <li>En Axapta, selecciona y copia los datos incluyendo los encabezados</li>
                        <li>Columnas requeridas: <strong>Código de artículo</strong></li>
                        <li>Columnas opcionales: <strong>Nombre del artículo</strong>, <strong>Estado</strong>, <strong>Grupo de artículos</strong>, <strong>Tipo de artículo</strong> y <strong>Precio</strong></li>
                        <li>Pega los datos en el área de texto de abajo (Ctrl+V)</li>
                        <li>El sistema actualizará productos existentes y creará los nuevos automáticamente</li>
                    </ol>
                </div>
                
                <div class="bg-amber-50 border border-amber-200 rounded-lg p-4 mb-4">
                    <h4 class="font-semibold text-amber-800 mb-2">
                        <i class="fas fa-magic mr-2"></i>Mapeo Automático
                    </h4>
                    <div class="text-sm text-amber-700">
                        <p class="mb-2">El sistema mapeará automáticamente:</p>
                        <ul class="list-disc list-inside space-y-1">
                            <li><strong>Grupos de artículos</strong> → Categorías según configuración</li>
                            <li><strong>Estado 99</strong> → Producto descontinuado (rojo)</li>
                            <li><strong>Estado &lt; 99</strong> → Producto activo (verde)</li>
                            <li><strong>URLs de imágenes</strong> → Se generan automáticamente</li>
                        </ul>
                        <a href="index.php?page=gestionar_categorias" class="text-amber-900 underline font-semibold mt-2 inline-block">
                            Gestionar categorías →
                        </a>
                    </div>
                </div>
            </div>
            
            <label class="block text-sm font-medium text-gray-700 mb-2">
                Pega aquí los datos copiados desde Axapta:
            </label>
            <textarea name="paste_data" 
                      class="w-full h-64 p-4 border-2 border-gray-300 rounded-lg font-mono text-sm focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                      placeholder="Código de artículo&#9;Nombre del artículo&#9;Estado&#9;Grupo de artículos&#9;Tipo de artículo&#9;Precio..."
                      required></textarea>
            
            <div class="mt-4 grid grid-cols-2 gap-4">
                <div class="bg-gray-50 p-4 rounded-lg">
                    <h5 class="font-semibold text-gray-700 mb-2">Columnas esperadas:</h5>
                    <ul class="text-xs text-gray-600 space-y-1">
                        <li>• <strong>Código de artículo</strong> (Requerido)</li>
                        <li>• Nombre del artículo</li>
                        <li>• Estado</li>
                        <li>• Grupo de artículos</li>
                        <li>• Tipo de artículo</li>
                        <li>• Precio</li>
                    </ul>
                </div>
                <div class="bg-gray-50 p-4 rounded-lg">
                    <h5 class="font-semibold text-gray-700 mb-2">Formato de datos:</h5>
                    <ul class="text-xs text-gray-600 space-y-1">
                        <li>• Delimitador: Tabulador</li>
                        <li>• Decimales: Coma o punto</li>
                        <li>• Miles: Punto (se eliminará)</li>
                        <li>• Primera línea: Encabezados</li>
                        <li>• Estado: Numérico (99 = descontinuado)</li>
                    </ul>
                </div>
            </div>
            
            <div class="flex justify-end gap-4 mt-6 pt-6 border-t">
                <button type="button" onclick="closeModal('import-modal')" 
                        class="px-6 py-3 bg-gray-200 text-gray-800 rounded-lg hover:bg-gray-300 font-semibold">
                    Cancelar
                </button>
                <button type="submit" 
                        class="px-6 py-3 bg-gradient-to-r from-purple-600 to-indigo-700 text-white rounded-lg hover:from-purple-700 hover:to-indigo-800 font-semibold">
                    <i class="fas fa-upload mr-2"></i>Procesar Importación
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Modal de Eliminación -->
<div id="delete-modal" class="fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center p-4 hidden z-50">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-md p-6">
        <h2 class="text-2xl font-bold text-gray-800 mb-4">Confirmar Eliminación</h2>
        <p class="text-gray-600 mb-6">¿Estás seguro de que deseas eliminar el producto <strong id="delete-ref-display"></strong>? Esta acción no se puede deshacer.</p>
        <form action="index.php?page=importar_productos" method="POST">
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="referencia" id="delete-ref-input">
            <div class="flex justify-end gap-4">
                <button type="button" onclick="closeModal('delete-modal')" class="px-4 py-2 bg-gray-200 text-gray-800 rounded-lg hover:bg-gray-300">Cancelar</button>
                <button type="submit" class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">Eliminar</button>
            </div>
        </form>
    </div>
</div>

<!-- Modal de Creación MEJORADO -->
<div id="create-modal" class="fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center p-4 hidden z-50">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-2xl p-6 max-h-[90vh] overflow-y-auto">
        <h2 class="text-2xl font-bold text-gray-800 mb-4">Añadir Nuevo Producto</h2>
        <form action="index.php?page=importar_productos" method="POST" class="space-y-4">
            <input type="hidden" name="action" value="create">
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700">Referencia *</label>
                    <input type="text" name="referencia" id="new-referencia" 
                           class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-purple-500" 
                           required onkeyup="updateImagePreview()">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700">Estado</label>
                    <input type="text" name="estado" class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-purple-500" 
                           placeholder="99 = Descontinuado">
                </div>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700">Nombre *</label>
                <input type="text" name="nombre" class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-purple-500" required>
            </div>
            
            <div class="bg-gray-50 p-4 rounded-lg">
                <label class="block text-sm font-medium text-gray-700 mb-2">Vista previa de imagen</label>
                <div class="flex items-center gap-4">
                    <img id="preview-image" 
                         class="h-20 w-20 object-cover rounded-lg border-2 border-gray-300"
                         src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='80' height='80' viewBox='0 0 24 24'%3E%3Cpath fill='%23ccc' d='M21 19V5c0-1.1-.9-2-2-2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2zM8.5 13.5l2.5 3.01L14.5 12l4.5 6H5l3.5-4.5z'/%3E%3C/svg%3E"
                        alt="Vista previa">
                   <div class="text-sm text-gray-600">
                       <p>URL de imagen:</p>
                       <p id="preview-url" class="font-mono text-xs text-blue-600"></p>
                       <p class="text-xs text-gray-500 mt-1">La imagen se generará automáticamente con la referencia</p>
                   </div>
               </div>
           </div>
           
           <div class="grid grid-cols-2 gap-4">
               <div>
                   <label class="block text-sm font-medium text-gray-700">Grupo de Artículo (Axapta)</label>
                   <input type="text" name="grupo_articulo" class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-purple-500" 
                          placeholder="Ej: SOFA, MESA CENTRO, 050...">
                   <p class="text-xs text-gray-500 mt-1">Se asignarán las categorías automáticamente</p>
               </div>
               <div>
                   <label class="block text-sm font-medium text-gray-700">Tipo de Artículo</label>
                   <select name="tipo_articulo" class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-purple-500">
                       <option value="Artículo">Artículo</option>
                       <option value="L.MAT">L.MAT</option>
                   </select>
               </div>
           </div>
           
           <div class="grid grid-cols-2 gap-4">
               <div>
                   <label class="block text-sm font-medium text-gray-700">Categoría Principal</label>
                   <select name="categoria1_id" id="create-cat1" class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-purple-500">
                       <option value="">Automático según grupo</option>
                       <?php foreach($categorias1 as $cat): ?>
                           <option value="<?php echo $cat['id']; ?>">
                               <?php echo htmlspecialchars($cat['nombre']); ?>
                           </option>
                       <?php endforeach; ?>
                   </select>
               </div>
               <div>
                   <label class="block text-sm font-medium text-gray-700">Subcategoría</label>
                   <select name="categoria2_id" id="create-cat2" class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-purple-500">
                       <option value="">Automático según grupo</option>
                   </select>
               </div>
           </div>
           
           <div>
               <label class="block text-sm font-medium text-gray-700">Precio</label>
               <input type="number" step="0.01" name="precio" class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-purple-500" 
                      placeholder="0.00">
           </div>
           
           <div class="flex justify-end gap-4 pt-4 border-t">
               <button type="button" onclick="closeModal('create-modal')" 
                       class="px-4 py-2 bg-gray-200 text-gray-800 rounded-lg hover:bg-gray-300">
                   Cancelar
               </button>
               <button type="submit" 
                       class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                   Crear Producto
               </button>
           </div>
       </form>
   </div>
</div>

<script>
// Datos de categorías para JavaScript
const categorias2Data = <?php echo json_encode($categorias2_full); ?>;

function openModal(modalId) {
   document.getElementById(modalId).classList.remove('hidden');
}

function closeModal(modalId) {
   document.getElementById(modalId).classList.add('hidden');
}

function openDeleteModal(referencia) {
   document.getElementById('delete-ref-display').innerText = referencia;
   document.getElementById('delete-ref-input').value = referencia;
   openModal('delete-modal');
}

function updateImagePreview() {
   const referencia = document.getElementById('new-referencia').value;
   const previewUrl = document.getElementById('preview-url');
   const previewImage = document.getElementById('preview-image');
   
   if (referencia) {
       const imageUrl = '<?php echo IMAGE_BASE_URL; ?>' + referencia + '.jpg';
       previewUrl.textContent = imageUrl;
       previewImage.src = imageUrl;
       previewImage.onerror = function() {
           this.src = 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'80\' height=\'80\' viewBox=\'0 0 24 24\'%3E%3Cpath fill=\'%23ccc\' d=\'M21 19V5c0-1.1-.9-2-2-2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2zM8.5 13.5l2.5 3.01L14.5 12l4.5 6H5l3.5-4.5z\'/%3E%3C/svg%3E';
       };
   } else {
       previewUrl.textContent = 'Ingrese una referencia...';
       previewImage.src = 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'80\' height=\'80\' viewBox=\'0 0 24 24\'%3E%3Cpath fill=\'%23ccc\' d=\'M21 19V5c0-1.1-.9-2-2-2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2zM8.5 13.5l2.5 3.01L14.5 12l4.5 6H5l3.5-4.5z\'/%3E%3C/svg%3E';
   }
}

// Filtrar subcategorías según categoría principal seleccionada
function updateSubcategories(cat1Select, cat2Select) {
   const selectedCat1 = cat1Select.value;
   const cat2Options = cat2Select.querySelectorAll('option');
   
   // Mostrar todas las opciones primero
   cat2Options.forEach(option => {
       if (option.value === '') return; // Skip la opción vacía
       
       const optionCat1 = option.getAttribute('data-cat1');
       if (!selectedCat1 || !optionCat1 || optionCat1 === selectedCat1) {
           option.style.display = '';
       } else {
           option.style.display = 'none';
       }
   });
}

// Event listeners para filtros
document.getElementById('filter-cat1').addEventListener('change', function() {
   updateSubcategories(this, document.getElementById('filter-cat2'));
});

// Para el modal de creación
document.getElementById('create-cat1').addEventListener('change', function() {
   const cat2Select = document.getElementById('create-cat2');
   const selectedCat1Id = this.value;
   
   // Limpiar opciones actuales
   cat2Select.innerHTML = '<option value="">Automático según grupo</option>';
   
   if (selectedCat1Id) {
       // Filtrar y agregar subcategorías correspondientes
       const filteredCat2 = categorias2Data.filter(cat => {
           // Necesitamos encontrar el id de la categoría padre
           const cat1Option = Array.from(this.options).find(opt => opt.value == selectedCat1Id);
           if (cat1Option) {
               return cat.cat1_nombre === cat1Option.textContent.trim();
           }
           return false;
       });
       
       filteredCat2.forEach(cat => {
           const option = document.createElement('option');
           option.value = cat.id;
           option.textContent = cat.nombre;
           cat2Select.appendChild(option);
       });
   }
});

// Manejar checkbox de auto URL
document.addEventListener('change', function(e) {
   if (e.target.name === 'auto_url') {
       const form = e.target.closest('form');
       const urlInput = form.querySelector('input[name="url_imagen"]');
       if (e.target.checked) {
           const referencia = form.querySelector('input[name="referencia"]').value;
           urlInput.value = '<?php echo IMAGE_BASE_URL; ?>' + referencia + '.jpg';
       }
   }
});

// Cerrar modales al hacer clic fuera
document.addEventListener('click', function(event) {
   if (event.target.classList.contains('bg-opacity-50')) {
       event.target.classList.add('hidden');
   }
});

// Prevenir cierre al hacer clic dentro del modal
document.querySelectorAll('.bg-white').forEach(modal => {
   modal.addEventListener('click', function(event) {
       event.stopPropagation();
   });
});

// Filtro de estado especial para "activos"
document.querySelector('select[name="estado"]').addEventListener('change', function() {
   if (this.value === 'activo') {
       // Aquí podrías agregar lógica especial para filtrar estados < 99
   }
});
</script>

<style>
.animate-fade-in {
   animation: fadeIn 0.5s ease-in;
}
.animate-slide-up {
   animation: slideUp 0.5s ease-out;
}
@keyframes fadeIn {
   from { opacity: 0; transform: translateY(-10px); }
   to { opacity: 1; transform: translateY(0); }
}
@keyframes slideUp {
   from { opacity: 0; transform: translateY(20px); }
   to { opacity: 1; transform: translateY(0); }
}

/* Estilo para imágenes con error */
img[onerror] {
   position: relative;
}

/* Hover effect para imágenes */
.group:hover img {
   transform: scale(1.05);
   transition: transform 0.2s;
}

/* Hacer que los selects con opciones ocultas se vean mejor */
select option[style*="display: none"] {
   display: none !important;
}
</style>