<?php
// pages/stock_calculado.php

// Incluir las funciones
require_once __DIR__ . '/stock_calculado_functions.php';

date_default_timezone_set('America/Bogota');

// --- LÓGICA DE VISUALIZACIÓN ---
$db_error = null;
$paginated_items = [];
$total_items = 0;
$total_pages = 1;
$current_page_num = 1;
$kpi_total_units = 0;
$kpi_buildable_lmats = 0;
$kpi_total_articulos_vendibles = 0;
$kpi_total_imperfectos = 0;
$kpi_total_excepciones = 0;
$bodega_options = [];
$almacen_options = [];
$categoria_options = [];

$upload_success_message = $_SESSION['upload_success'] ?? null;
unset($_SESSION['upload_success']);
$upload_error_message = $_SESSION['upload_error'] ?? null;
unset($_SESSION['upload_error']);

try {
    $generationResult = generate_combinations($pdo);
    $allCombinations = $generationResult['combinations'];
    
    // Obtener estadísticas generales
    $stats = get_stock_statistics($allCombinations);
    
    // Opciones para filtros
    $bodega_options = array_keys($stats['bodegas']);
    sort($bodega_options);
    $almacen_options = array_keys($stats['almacenes']);
    sort($almacen_options);
    $categoria_options = array_keys($stats['categorias']);
    sort($categoria_options);
    
    // KPIs principales
    $kpi_total_units = $stats['total_units'];
    $kpi_buildable_lmats = $stats['total_lmats'];
    $kpi_total_articulos_vendibles = $stats['total_articulos'];
    $kpi_total_imperfectos = $stats['total_imperfectos'];
    $kpi_total_excepciones = $stats['total_excepciones'];
    
    // Aplicar filtros
    $filteredCombinations = filter_and_sort($allCombinations, $_GET);
    $total_items = count($filteredCombinations);
    
    // Paginación
    $items_per_page = 50;
    $current_page_num = isset($_GET['page_num']) ? max(1, intval($_GET['page_num'])) : 1;
    $total_pages = max(1, ceil($total_items / $items_per_page));
    
    if ($current_page_num > $total_pages) {
        $current_page_num = $total_pages;
    }
    
    $offset = ($current_page_num - 1) * $items_per_page;
    $paginated_items = array_slice($filteredCombinations, $offset, $items_per_page);
    
} catch (Exception $e) {
    $db_error = $e->getMessage();
}

// Recuperar valores de filtros
$search_get = $_GET['search'] ?? '';
$show_type_get = $_GET['show_type'] ?? 'todos';
$sort_get = $_GET['sort'] ?? 'alpha_asc';
$bodegas_get = $_GET['bodegas'] ?? [];
$almacenes_get = $_GET['almacenes'] ?? [];
$qty_get = $_GET['qty'] ?? 0;
$qty_op_get = $_GET['qty_op'] ?? '>=';
$precio_get = $_GET['precio'] ?? 0;
$precio_op_get = $_GET['precio_op'] ?? '>=';
$descuento_get = $_GET['descuento'] ?? 0;
$descuento_op_get = $_GET['descuento_op'] ?? '>=';
$precio_final_get = $_GET['precio_final'] ?? 0;
$precio_final_op_get = $_GET['precio_final_op'] ?? '>=';
$categorias_get = $_GET['categorias'] ?? [];
$filtro_imagen_get = $_GET['filtro_imagen'] ?? 'todos';
$mostrar_excepciones_get = $_GET['mostrar_excepciones'] ?? 'todos';
$mostrar_imperfectos_get = $_GET['mostrar_imperfectos'] ?? 'todos';
?>

<div class="space-y-6">
    <!-- KPIs Header mejorado -->
    <header class="bg-white p-4 rounded-lg shadow-md">
        <div class="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-5 gap-4">
            <div class="bg-gray-50 p-3 rounded-lg flex items-center gap-3">
                <div class="bg-yellow-100 text-yellow-600 p-3 rounded-full">
                    <i class="fas fa-box-open"></i>
                </div>
                <div>
                    <p class="text-xs text-gray-500">Unidades</p>
                    <p class="text-xl font-bold text-gray-800"><?php echo number_format($kpi_total_units); ?></p>
                </div>
            </div>
            
            <div class="bg-gray-50 p-3 rounded-lg flex items-center gap-3">
                <div class="bg-purple-100 text-purple-600 p-3 rounded-full">
                    <i class="fas fa-hammer"></i>
                </div>
                <div>
                    <p class="text-xs text-gray-500">LMATs</p>
                    <p class="text-xl font-bold text-gray-800"><?php echo number_format($kpi_buildable_lmats); ?></p>
                </div>
            </div>
            
            <div class="bg-gray-50 p-3 rounded-lg flex items-center gap-3">
                <div class="bg-teal-100 text-teal-600 p-3 rounded-full">
                    <i class="fas fa-tag"></i>
                </div>
                <div>
                    <p class="text-xs text-gray-500">Artículos</p>
                    <p class="text-xl font-bold text-gray-800"><?php echo number_format($kpi_total_articulos_vendibles); ?></p>
                </div>
            </div>
            
            <div class="bg-gray-50 p-3 rounded-lg flex items-center gap-3">
                <div class="bg-red-100 text-red-600 p-3 rounded-full">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <div>
                    <p class="text-xs text-gray-500">Imperfectos</p>
                    <p class="text-xl font-bold text-gray-800"><?php echo number_format($kpi_total_imperfectos); ?></p>
                </div>
            </div>
            
            <div class="bg-gray-50 p-3 rounded-lg flex items-center gap-3">
                <div class="bg-orange-100 text-orange-600 p-3 rounded-full">
                    <i class="fas fa-puzzle-piece"></i>
                </div>
                <div>
                    <p class="text-xs text-gray-500">Excepciones</p>
                    <p class="text-xl font-bold text-gray-800"><?php echo number_format($kpi_total_excepciones); ?></p>
                </div>
            </div>
        </div>
    </header>

    <!-- Mensajes de éxito/error -->
    <?php if ($upload_success_message): ?>
    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative" role="alert">
        <span class="block sm:inline"><?php echo htmlspecialchars($upload_success_message); ?></span>
    </div>
    <?php endif; ?>

    <?php if ($upload_error_message): ?>
    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
        <span class="block sm:inline"><?php echo htmlspecialchars($upload_error_message); ?></span>
    </div>
    <?php endif; ?>

    <?php if ($db_error): ?>
    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
        <strong class="font-bold">Error de base de datos:</strong>
        <span class="block sm:inline"><?php echo htmlspecialchars($db_error); ?></span>
    </div>
    <?php endif; ?>

    <!-- Filtros mejorados -->
    <div class="bg-white p-6 rounded-lg shadow-md">
        <form method="GET" action="index.php" class="space-y-4" id="filterForm">
            <input type="hidden" name="page" value="stock_calculado">
            
            <!-- Título y botón de colapsar -->
            <div class="flex justify-between items-center">
                <h3 class="text-lg font-semibold text-gray-900">
                    <i class="fas fa-filter mr-2"></i>Filtros Avanzados
                </h3>
                <div class="flex items-center gap-4">
                    <span class="text-sm text-gray-600">
                        <i class="fas fa-database mr-1"></i>
                        <strong><?php echo number_format($total_items); ?></strong> resultados
                    </span>
                    <!-- Botones de vista -->
                    <div class="flex bg-gray-200 rounded-lg p-1">
                        <button type="button" onclick="setView('table')" id="viewTable" class="px-3 py-1 rounded text-sm font-medium bg-white text-gray-700 transition-colors">
                            <i class="fas fa-table"></i> Tabla
                        </button>
                        <button type="button" onclick="setView('cards')" id="viewCards" class="px-3 py-1 rounded text-sm font-medium text-gray-500 hover:text-gray-700 transition-colors">
                            <i class="fas fa-th"></i> Tarjetas
                        </button>
                    </div>
                    <button type="button" onclick="toggleFilters()" class="text-gray-500 hover:text-gray-700 transition-colors">
                        <i class="fas fa-chevron-down transition-transform duration-200" id="filterToggleIcon"></i>
                    </button>
                </div>
            </div>
            
            <div id="filterContent" class="hidden">
                <!-- Primera fila de filtros -->
                <div class="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-4 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Buscar</label>
                        <div class="relative">
                            <input type="text" name="search" value="<?php echo htmlspecialchars($search_get); ?>"
                                   placeholder="Referencia, nombre o código..."
                                   class="w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <i class="fas fa-search absolute left-3 top-3 text-gray-400"></i>
                        </div>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Tipo de Producto</label>
                        <select name="show_type" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="todos" <?php echo $show_type_get === 'todos' ? 'selected' : ''; ?>>🔷 Todos los tipos</option>
                            <option value="solo_lmats" <?php echo $show_type_get === 'solo_lmats' ? 'selected' : ''; ?>>📦 Solo LMATs (Kits)</option>
                            <option value="solo_articulos" <?php echo $show_type_get === 'solo_articulos' ? 'selected' : ''; ?>>🏷️ Solo Artículos</option>
                            <option value="solo_imperfectos" <?php echo $show_type_get === 'solo_imperfectos' ? 'selected' : ''; ?>>⚠️ Solo Imperfectos</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Filtrar Excepciones</label>
                        <select name="mostrar_excepciones" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="todos" <?php echo $mostrar_excepciones_get === 'todos' ? 'selected' : ''; ?>>🔸 Mostrar todas</option>
                            <option value="solo_excepciones" <?php echo $mostrar_excepciones_get === 'solo_excepciones' ? 'selected' : ''; ?>>🧩 Solo Excepciones (Partes)</option>
                            <option value="sin_excepciones" <?php echo $mostrar_excepciones_get === 'sin_excepciones' ? 'selected' : ''; ?>>✅ Sin Excepciones</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Productos Imperfectos</label>
                        <select name="mostrar_imperfectos" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="todos" <?php echo $mostrar_imperfectos_get === 'todos' ? 'selected' : ''; ?>>🔹 Todos</option>
                            <option value="solo_imperfectos" <?php echo $mostrar_imperfectos_get === 'solo_imperfectos' ? 'selected' : ''; ?>>🏷️ Solo con Código Único</option>
                            <option value="sin_imperfectos" <?php echo $mostrar_imperfectos_get === 'sin_imperfectos' ? 'selected' : ''; ?>>📦 Solo Productos Normales</option>
                        </select>
                    </div>
                </div>
                
                <!-- Segunda fila de filtros -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Ordenar por</label>
                        <select name="sort" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <optgroup label="📝 Alfabético">
                                <option value="alpha_asc" <?php echo $sort_get === 'alpha_asc' ? 'selected' : ''; ?>>A → Z</option>
                                <option value="alpha_desc" <?php echo $sort_get === 'alpha_desc' ? 'selected' : ''; ?>>Z → A</option>
                            </optgroup>
                            <optgroup label="💰 Precio">
                                <option value="price_desc" <?php echo $sort_get === 'price_desc' ? 'selected' : ''; ?>>Mayor → Menor</option>
                                <option value="price_asc" <?php echo $sort_get === 'price_asc' ? 'selected' : ''; ?>>Menor → Mayor</option>
                            </optgroup>
                            <optgroup label="📊 Cantidad">
                                <option value="qty_desc" <?php echo $sort_get === 'qty_desc' ? 'selected' : ''; ?>>Mayor → Menor</option>
                                <option value="qty_asc" <?php echo $sort_get === 'qty_asc' ? 'selected' : ''; ?>>Menor → Mayor</option>
                            </optgroup>
                            <optgroup label="⭐ Priorizar">
                                <option value="sin_foto_primero" <?php echo $sort_get === 'sin_foto_primero' ? 'selected' : ''; ?>>🖼️ Sin Foto Primero</option>
                                <option value="imperfectos_primero" <?php echo $sort_get === 'imperfectos_primero' ? 'selected' : ''; ?>>⚠️ Imperfectos Primero</option>
                                <option value="excepciones_primero" <?php echo $sort_get === 'excepciones_primero' ? 'selected' : ''; ?>>🧩 Excepciones Primero</option>
                            </optgroup>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Estado de Imágenes</label>
                        <select name="filtro_imagen" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="todos" <?php echo $filtro_imagen_get === 'todos' ? 'selected' : ''; ?>>📷 Todas</option>
                            <option value="con_foto" <?php echo $filtro_imagen_get === 'con_foto' ? 'selected' : ''; ?>>✅ Con Foto</option>
                            <option value="sin_foto" <?php echo $filtro_imagen_get === 'sin_foto' ? 'selected' : ''; ?>>❌ Sin Foto</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Tiendas</label>
                        <div class="relative">
                            <button type="button" onclick="toggleDropdown('bodegasDropdown')" class="w-full px-3 py-2 border border-gray-300 rounded-md bg-white text-left focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <span id="bodegasLabel">
                                    <?php 
                                    $count = count($bodegas_get);
                                    echo $count > 0 ? $count . ' seleccionadas' : 'Todas las tiendas';
                                    ?>
                                </span>
                                <i class="fas fa-chevron-down float-right mt-1"></i>
                            </button>
                            <div id="bodegasDropdown" class="hidden absolute z-10 mt-1 w-full bg-white border border-gray-300 rounded-md shadow-lg max-h-60 overflow-y-auto">
                                <div class="p-2">
                                    <label class="flex items-center p-2 hover:bg-gray-50 cursor-pointer">
                                        <input type="checkbox" onclick="toggleAll('bodegas')" class="mr-2">
                                        <span class="font-medium">Seleccionar todas</span>
                                    </label>
                                    <div class="border-t my-1"></div>
                                    <?php foreach ($bodega_options as $bodega): ?>
                                    <label class="flex items-center p-2 hover:bg-gray-50 cursor-pointer">
                                        <input type="checkbox" name="bodegas[]" value="<?php echo htmlspecialchars($bodega); ?>" 
                                               <?php echo in_array($bodega, $bodegas_get) ? 'checked' : ''; ?>
                                               onchange="updateLabel('bodegas')" class="bodega-checkbox mr-2">
                                        <span><?php echo htmlspecialchars($bodega); ?></span>
                                    </label>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Categorías</label>
                        <div class="relative">
                            <button type="button" onclick="toggleDropdown('categoriasDropdown')" class="w-full px-3 py-2 border border-gray-300 rounded-md bg-white text-left focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <span id="categoriasLabel">
                                    <?php 
                                    $count = count($categorias_get);
                                    echo $count > 0 ? $count . ' seleccionadas' : 'Todas las categorías';
                                    ?>
                                </span>
                                <i class="fas fa-chevron-down float-right mt-1"></i>
                            </button>
                            <div id="categoriasDropdown" class="hidden absolute z-10 mt-1 w-full bg-white border border-gray-300 rounded-md shadow-lg max-h-60 overflow-y-auto">
                                <div class="p-2">
                                    <label class="flex items-center p-2 hover:bg-gray-50 cursor-pointer">
                                        <input type="checkbox" onclick="toggleAll('categorias')" class="mr-2">
                                        <span class="font-medium">Seleccionar todas</span>
                                    </label>
                                    <div class="border-t my-1"></div>
                                    <?php foreach ($categoria_options as $categoria): ?>
                                    <label class="flex items-center p-2 hover:bg-gray-50 cursor-pointer">
                                        <input type="checkbox" name="categorias[]" value="<?php echo htmlspecialchars($categoria); ?>" 
                                               <?php echo in_array($categoria, $categorias_get) ? 'checked' : ''; ?>
                                               onchange="updateLabel('categorias')" class="categoria-checkbox mr-2">
                                        <span><?php echo htmlspecialchars($categoria); ?></span>
                                    </label>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Filtros numéricos con rangos -->
                <div class="border-t pt-4 mt-4">
                    <h4 class="text-sm font-medium text-gray-700 mb-3">Filtros Numéricos</h4>
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                        <div>
                            <label class="block text-xs text-gray-600 mb-1">Cantidad</label>
                            <div class="flex gap-2">
                                <select name="qty_op" class="px-2 py-2 border border-gray-300 rounded-md text-sm">
                                    <option value=">=" <?php echo $qty_op_get === '>=' ? 'selected' : ''; ?>>≥</option>
                                    <option value="<=" <?php echo $qty_op_get === '<=' ? 'selected' : ''; ?>>≤</option>
                                    <option value="=" <?php echo $qty_op_get === '=' ? 'selected' : ''; ?>>=</option>
                                </select>
                                <input type="number" name="qty" value="<?php echo $qty_get; ?>" 
                                       placeholder="Unidades" min="0"
                                       class="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm">
                            </div>
                        </div>
                        
                        <div>
                            <label class="block text-xs text-gray-600 mb-1">Precio Original</label>
                            <div class="flex gap-2">
                                <select name="precio_op" class="px-2 py-2 border border-gray-300 rounded-md text-sm">
                                    <option value=">=" <?php echo $precio_op_get === '>=' ? 'selected' : ''; ?>>≥</option>
                                    <option value="<=" <?php echo $precio_op_get === '<=' ? 'selected' : ''; ?>>≤</option>
                                    <option value="=" <?php echo $precio_op_get === '=' ? 'selected' : ''; ?>>=</option>
                                </select>
                                <input type="number" name="precio" value="<?php echo $precio_get; ?>" 
                                       placeholder="$0.00" min="0" step="0.01"
                                       class="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm">
                            </div>
                        </div>
                        
                        <div>
                            <label class="block text-xs text-gray-600 mb-1">Descuento %</label>
                            <div class="flex gap-2">
                                <select name="descuento_op" class="px-2 py-2 border border-gray-300 rounded-md text-sm">
                                    <option value=">=" <?php echo $descuento_op_get === '>=' ? 'selected' : ''; ?>>≥</option>
                                    <option value="<=" <?php echo $descuento_op_get === '<=' ? 'selected' : ''; ?>>≤</option>
                                    <option value="=" <?php echo $descuento_op_get === '=' ? 'selected' : ''; ?>>=</option>
                                </select>
                                <input type="number" name="descuento" value="<?php echo $descuento_get; ?>" 
                                       placeholder="0%" min="0" max="100"
                                       class="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm">
                            </div>
                        </div>
                        
                        <div>
                            <label class="block text-xs text-gray-600 mb-1">Precio Final</label>
                            <div class="flex gap-2">
                                <select name="precio_final_op" class="px-2 py-2 border border-gray-300 rounded-md text-sm">
                                    <option value=">=" <?php echo $precio_final_op_get === '>=' ? 'selected' : ''; ?>>≥</option>
                                    <option value="<=" <?php echo $precio_final_op_get === '<=' ? 'selected' : ''; ?>>≤</option>
                                    <option value="=" <?php echo $precio_final_op_get === '=' ? 'selected' : ''; ?>>=</option>
                                </select>
                                <input type="number" name="precio_final" value="<?php echo $precio_final_get; ?>" 
                                       placeholder="$0.00" min="0" step="0.01"
                                       class="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm">
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Botones de acción -->
                <div class="flex justify-between items-center pt-4">
                    <div class="flex gap-2">
                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors">
                            <i class="fas fa-search mr-2"></i>Aplicar Filtros
                        </button>
                        <a href="index.php?page=stock_calculado" class="px-4 py-2 bg-gray-600 text-white rounded-md hover:bg-gray-700 transition-colors">
                            <i class="fas fa-redo mr-2"></i>Limpiar Todo
                        </a>
                        <button type="button" onclick="toggleQuickFilters()" class="px-4 py-2 bg-purple-600 text-white rounded-md hover:bg-purple-700 transition-colors">
                            <i class="fas fa-sliders-h mr-2"></i>Filtros Rápidos
                        </button>
                    </div>
                </div>
                
                <!-- Filtros Rápidos (ocultos por defecto) -->
                <div id="quickFilters" class="hidden mt-4 pt-4 border-t">
                    <h4 class="text-sm font-semibold text-gray-700 mb-3">🚀 Filtros Rápidos - Accesos Directos</h4>
                    <div class="flex flex-wrap gap-2">
                        <button onclick="applyQuickFilter('con_descuento')" class="px-3 py-1 bg-green-100 text-green-700 rounded-full hover:bg-green-200 text-sm transition-colors">
                            <i class="fas fa-tag mr-1"></i>Con Descuento
                        </button>
                        <button onclick="applyQuickFilter('sin_foto')" class="px-3 py-1 bg-red-100 text-red-700 rounded-full hover:bg-red-200 text-sm transition-colors">
                            <i class="fas fa-image mr-1"></i>Sin Foto
                        </button>
                        <button onclick="applyQuickFilter('imperfectos')" class="px-3 py-1 bg-yellow-100 text-yellow-700 rounded-full hover:bg-yellow-200 text-sm transition-colors">
                            <i class="fas fa-exclamation-triangle mr-1"></i>Imperfectos
                        </button>
                        <button onclick="applyQuickFilter('partes')" class="px-3 py-1 bg-orange-100 text-orange-700 rounded-full hover:bg-orange-200 text-sm transition-colors">
                            <i class="fas fa-puzzle-piece mr-1"></i>Solo Partes
                        </button>
                        <button onclick="applyQuickFilter('stock_alto')" class="px-3 py-1 bg-blue-100 text-blue-700 rounded-full hover:bg-blue-200 text-sm transition-colors">
                            <i class="fas fa-boxes mr-1"></i>Stock Alto (>10)
                        </button>
                        <button onclick="applyQuickFilter('stock_bajo')" class="px-3 py-1 bg-purple-100 text-purple-700 rounded-full hover:bg-purple-200 text-sm transition-colors">
                            <i class="fas fa-exclamation mr-1"></i>Stock Bajo (≤3)
                        </button>
                        <button onclick="applyQuickFilter('precio_alto')" class="px-3 py-1 bg-indigo-100 text-indigo-700 rounded-full hover:bg-indigo-200 text-sm transition-colors">
                            <i class="fas fa-dollar-sign mr-1"></i>Precio >$100k
                        </button>
                        <button onclick="applyQuickFilter('solo_lmats')" class="px-3 py-1 bg-pink-100 text-pink-700 rounded-full hover:bg-pink-200 text-sm transition-colors">
                            <i class="fas fa-cubes mr-1"></i>Solo Kits
                        </button>
                    </div>
                </div>
            </div>
        </form>
    </div>

    <!-- Tabla de resultados mejorada -->
    <div id="tableView" class="bg-white rounded-lg shadow-md overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            <input type="checkbox" id="select-all" class="rounded border-gray-300">
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Imagen</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Referencia</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nombre</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tipo</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Etiqueta</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Cantidad</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tienda</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ubicación</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Categoría</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Precio</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Desc.</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">P. Final</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Acciones</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($paginated_items as $item): ?>
                    <tr class="hover:bg-gray-50 <?php echo isset($item['es_imperfecto']) ? 'bg-yellow-50' : ''; ?>">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <input type="checkbox" name="selected_items[]" value="<?php echo htmlspecialchars($item['id']); ?>" 
                                   class="item-checkbox rounded border-gray-300">
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <?php if (!empty($item['url_imagen'])): ?>
                                <img src="<?php echo htmlspecialchars($item['url_imagen']); ?>" 
                                     alt="<?php echo htmlspecialchars($item['lmat']); ?>"
                                     class="h-12 w-12 object-cover rounded cursor-pointer hover:scale-110 transition-transform"
                                     onclick="openImageModal('<?php echo htmlspecialchars($item['url_imagen']); ?>')">
                            <?php else: ?>
                                <div class="h-12 w-12 bg-gray-200 rounded flex items-center justify-center">
                                    <i class="fas fa-image text-gray-400"></i>
                                </div>
                            <?php endif; ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap font-medium text-gray-900">
                            <?php echo htmlspecialchars($item['lmat']); ?>
                        </td>
                        <td class="px-6 py-4 text-sm text-gray-900">
                            <?php echo htmlspecialchars($item['nombre']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <?php 
                            $type_badges = [
                                'lmat' => '<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-purple-100 text-purple-800">LMAT</span>',
                                'articulo' => '<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800">Artículo</span>',
                                'imperfecto' => '<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 text-red-800">Imperfecto</span>'
                            ];
                            echo $type_badges[$item['type']] ?? '<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-gray-800">' . htmlspecialchars($item['type']) . '</span>';
                            ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <?php if (!empty($item['etiqueta'])): ?>
                                <?php if ($item['etiqueta'] === 'Parte'): ?>
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-orange-100 text-orange-800">
                                        <i class="fas fa-puzzle-piece mr-1"></i>Parte
                                    </span>
                                <?php elseif (isset($item['codigo_unico'])): ?>
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800">
                                        <i class="fas fa-barcode mr-1"></i><?php echo htmlspecialchars($item['codigo_unico']); ?>
                                    </span>
                                <?php else: ?>
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-gray-800">
                                        <?php echo htmlspecialchars($item['etiqueta']); ?>
                                    </span>
                                <?php endif; ?>
                            <?php endif; ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            <?php echo number_format($item['units']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            <?php echo htmlspecialchars($item['bodega']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            <?php echo htmlspecialchars($item['almacen']); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            <?php 
                            $categoria_display = $item['categoria1'] ?? $item['categoria'] ?? 'Sin categoría';
                            if (!empty($item['categoria2']) && $item['categoria2'] !== 'General') {
                                $categoria_display .= ' / ' . $item['categoria2'];
                            }
                            echo htmlspecialchars($categoria_display);
                            ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            $<?php echo number_format($item['precio'], 2); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            <?php if ($item['descuento'] > 0): ?>
                                <span class="text-green-600 font-semibold"><?php echo $item['descuento']; ?>%</span>
                            <?php else: ?>
                                -
                            <?php endif; ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-900">
                            $<?php echo number_format($item['precio_final'], 2); ?>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                            <div class="flex gap-2">
                                <?php if (empty($item['url_imagen'])): ?>
                                <form action="pages/actions/upload_image.php" method="POST" enctype="multipart/form-data" class="inline">
                                    <input type="hidden" name="referencia" value="<?php echo htmlspecialchars($item['lmat']); ?>">
                                    <input type="file" name="imagen" accept="image/*" required class="hidden" 
                                           id="file-<?php echo htmlspecialchars($item['id']); ?>"
                                           onchange="this.form.submit()">
                                    <label for="file-<?php echo htmlspecialchars($item['id']); ?>" 
                                           class="cursor-pointer text-blue-600 hover:text-blue-900">
                                        <i class="fas fa-camera"></i>
                                    </label>
                                </form>
                                <?php endif; ?>
                                
                                <?php if (isset($item['es_imperfecto']) && !empty($item['notas'])): ?>
                                <button onclick="showNotes('<?php echo htmlspecialchars($item['notas'], ENT_QUOTES); ?>')" 
                                        class="text-gray-600 hover:text-gray-900">
                                    <i class="fas fa-sticky-note"></i>
                                </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Paginación -->
        <?php if ($total_pages > 1): ?>
        <div class="bg-white px-4 py-3 flex items-center justify-between border-t border-gray-200 sm:px-6">
            <div class="flex-1 flex justify-between sm:hidden">
                <?php if ($current_page_num > 1): ?>
                <a href="?page=stock_calculado&page_num=<?php echo $current_page_num - 1; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                   class="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                    Anterior
                </a>
                <?php endif; ?>
                <?php if ($current_page_num < $total_pages): ?>
                <a href="?page=stock_calculado&page_num=<?php echo $current_page_num + 1; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                   class="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                    Siguiente
                </a>
                <?php endif; ?>
            </div>
            <div class="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                <div>
                    <p class="text-sm text-gray-700">
                        Mostrando <span class="font-medium"><?php echo ($offset + 1); ?></span> a 
                        <span class="font-medium"><?php echo min($offset + $items_per_page, $total_items); ?></span> de 
                        <span class="font-medium"><?php echo $total_items; ?></span> resultados
                    </p>
                </div>
                <div>
                    <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                        <!-- Botón Primera página -->
                        <?php if ($current_page_num > 1): ?>
                        <a href="?page=stock_calculado&page_num=1&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                            <i class="fas fa-angle-double-left"></i>
                        </a>
                        <?php endif; ?>
                        
                        <!-- Botón Anterior -->
                        <?php if ($current_page_num > 1): ?>
                        <a href="?page=stock_calculado&page_num=<?php echo $current_page_num - 1; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="relative inline-flex items-center px-2 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                            <i class="fas fa-angle-left"></i>
                        </a>
                        <?php endif; ?>
                        
                        <!-- Números de página -->
                        <?php
                        $start_page = max(1, $current_page_num - 4);
                        $end_page = min($total_pages, $current_page_num + 5);
                        
                        if ($start_page > 1): ?>
                            <span class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700">...</span>
                        <?php endif;
                        
                        for ($i = $start_page; $i <= $end_page; $i++): ?>
                        <a href="?page=stock_calculado&page_num=<?php echo $i; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="<?php echo $i === $current_page_num ? 'z-10 bg-indigo-50 border-indigo-500 text-indigo-600' : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'; ?> relative inline-flex items-center px-4 py-2 border text-sm font-medium">
                            <?php echo $i; ?>
                        </a>
                        <?php endfor;
                        
                        if ($end_page < $total_pages): ?>
                            <span class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700">...</span>
                        <?php endif; ?>
                        
                        <!-- Botón Siguiente -->
                        <?php if ($current_page_num < $total_pages): ?>
                        <a href="?page=stock_calculado&page_num=<?php echo $current_page_num + 1; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="relative inline-flex items-center px-2 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                            <i class="fas fa-angle-right"></i>
                        </a>
                        <?php endif; ?>
                        
                        <!-- Botón Última página -->
                        <?php if ($current_page_num < $total_pages): ?>
                        <a href="?page=stock_calculado&page_num=<?php echo $total_pages; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                            <i class="fas fa-angle-double-right"></i>
                        </a>
                        <?php endif; ?>
                    </nav>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <!-- Vista de Tarjetas -->
    <div id="cardsView" class="hidden">
        <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-4">
            <?php foreach ($paginated_items as $item): ?>
            <div class="bg-white rounded-lg shadow-md hover:shadow-xl transition-shadow overflow-hidden <?php echo isset($item['es_imperfecto']) ? 'ring-2 ring-yellow-400' : ''; ?>">
                <!-- Checkbox de selección -->
                <div class="absolute top-2 left-2 z-10">
                    <input type="checkbox" name="selected_items[]" value="<?php echo htmlspecialchars($item['id']); ?>" 
                           class="item-checkbox-card rounded border-gray-300 bg-white">
                </div>
                
                <!-- Imagen del producto -->
                <div class="relative h-48 bg-gray-100">
                    <?php if (!empty($item['url_imagen'])): ?>
                        <img src="<?php echo htmlspecialchars($item['url_imagen']); ?>" 
                             alt="<?php echo htmlspecialchars($item['lmat']); ?>"
                             class="w-full h-full object-cover cursor-pointer"
                             onclick="openImageModal('<?php echo htmlspecialchars($item['url_imagen']); ?>')">
                    <?php else: ?>
                        <div class="w-full h-full flex items-center justify-center">
                            <div class="text-center">
                                <i class="fas fa-image text-4xl text-gray-400 mb-2"></i>
                                <form action="pages/actions/upload_image.php" method="POST" enctype="multipart/form-data" class="inline">
                                    <input type="hidden" name="referencia" value="<?php echo htmlspecialchars($item['lmat']); ?>">
                                    <input type="file" name="imagen" accept="image/*" required class="hidden" 
                                           id="file-card-<?php echo htmlspecialchars($item['id']); ?>"
                                           onchange="this.form.submit()">
                                    <label for="file-card-<?php echo htmlspecialchars($item['id']); ?>" 
                                           class="cursor-pointer text-blue-600 hover:text-blue-700 text-sm font-medium">
                                        <i class="fas fa-camera mr-1"></i>Subir foto
                                    </label>
                                </form>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Badges de tipo y etiqueta -->
                    <div class="absolute top-2 right-2 flex flex-col gap-1">
                        <?php if ($item['type'] === 'lmat'): ?>
                            <span class="px-2 py-1 text-xs font-semibold rounded-full bg-purple-100 text-purple-800">LMAT</span>
                        <?php elseif ($item['type'] === 'articulo'): ?>
                            <span class="px-2 py-1 text-xs font-semibold rounded-full bg-blue-100 text-blue-800">Artículo</span>
                        <?php elseif ($item['type'] === 'imperfecto'): ?>
                            <span class="px-2 py-1 text-xs font-semibold rounded-full bg-red-100 text-red-800">Imperfecto</span>
                        <?php endif; ?>
                        
                        <?php if (!empty($item['etiqueta'])): ?>
                            <?php if ($item['etiqueta'] === 'Parte'): ?>
                                <span class="px-2 py-1 text-xs font-semibold rounded-full bg-orange-100 text-orange-800">
                                    <i class="fas fa-puzzle-piece mr-1"></i>Parte
                                </span>
                            <?php elseif (isset($item['codigo_unico'])): ?>
                                <span class="px-2 py-1 text-xs font-semibold rounded-full bg-yellow-100 text-yellow-800">
                                    <i class="fas fa-barcode mr-1"></i><?php echo htmlspecialchars($item['codigo_unico']); ?>
                                </span>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Badge de descuento -->
                    <?php if ($item['descuento'] > 0): ?>
                    <div class="absolute bottom-2 left-2">
                        <span class="px-2 py-1 text-xs font-bold rounded bg-green-500 text-white">
                            -<?php echo $item['descuento']; ?>%
                        </span>
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- Información del producto -->
                <div class="p-4">
                    <!-- Referencia y nombre -->
                    <h3 class="font-bold text-gray-900 text-sm mb-1 truncate">
                        <?php echo htmlspecialchars($item['lmat']); ?>
                    </h3>
                    <p class="text-gray-600 text-xs mb-2 truncate">
                        <?php echo htmlspecialchars($item['nombre'] ?: 'Sin nombre'); ?>
                    </p>
                    
                    <!-- Categoría -->
                    <p class="text-gray-500 text-xs mb-2">
                        <i class="fas fa-folder mr-1"></i>
                        <?php 
                        $cat = $item['categoria1'] ?? $item['categoria'] ?? 'Sin categoría';
                        echo htmlspecialchars(substr($cat, 0, 20));
                        if (strlen($cat) > 20) echo '...';
                        ?>
                    </p>
                    
                    <!-- Ubicación y cantidad -->
                    <div class="flex justify-between items-center mb-2 text-xs">
                        <span class="text-gray-600">
                            <i class="fas fa-map-marker-alt mr-1"></i>
                            <?php echo htmlspecialchars($item['bodega']); ?>
                        </span>
                        <span class="font-semibold text-gray-900">
                            <?php echo number_format($item['units']); ?> uds
                        </span>
                    </div>
                    
                    <!-- Precios mejorados -->
                    <div class="border-t pt-2 mt-2">
                        <?php if ($item['descuento'] > 0): ?>
                        <!-- Con descuento -->
                        <div class="space-y-1">
                            <div class="flex justify-between items-center">
                                <span class="text-xs text-gray-500">Precio:</span>
                                <span class="text-sm line-through text-gray-400">
                                    $<?php echo number_format($item['precio'], 2); ?>
                                </span>
                            </div>
                            <div class="flex justify-between items-center">
                                <span class="text-xs text-gray-500">Descuento:</span>
                                <span class="text-sm font-semibold text-green-600">
                                    -<?php echo $item['descuento']; ?>%
                                </span>
                            </div>
                            <div class="flex justify-between items-center bg-green-50 -mx-4 px-4 py-1 rounded">
                                <span class="text-xs font-semibold text-green-700">Precio Final:</span>
                                <span class="text-lg font-bold text-green-700">
                                    $<?php echo number_format($item['precio_final'], 2); ?>
                                </span>
                            </div>
                            <div class="text-center">
                                <span class="text-xs text-green-600">
                                    Ahorras $<?php echo number_format($item['precio'] - $item['precio_final'], 2); ?>
                                </span>
                            </div>
                        </div>
                        <?php else: ?>
                        <!-- Sin descuento -->
                        <div class="space-y-1">
                            <div class="flex justify-between items-center">
                                <span class="text-xs text-gray-500">Precio:</span>
                                <span class="text-lg font-bold text-gray-900">
                                    $<?php echo number_format($item['precio_final'], 2); ?>
                                </span>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <!-- Precio por unidad si hay múltiples -->
                        <?php if ($item['units'] > 1): ?>
                        <div class="text-center mt-1 pt-1 border-t">
                            <span class="text-xs text-gray-500">
                                $<?php echo number_format($item['precio_final'] / $item['units'], 2); ?> por unidad
                            </span>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Botón de notas para imperfectos -->
                    <?php if (isset($item['es_imperfecto']) && !empty($item['notas'])): ?>
                    <div class="mt-2 pt-2 border-t">
                        <button onclick="showNotes('<?php echo htmlspecialchars($item['notas'], ENT_QUOTES); ?>')" 
                                class="w-full text-xs text-gray-600 hover:text-gray-900 flex items-center justify-center">
                            <i class="fas fa-sticky-note mr-1"></i> Ver notas
                        </button>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Paginación para vista de tarjetas -->
        <?php if ($total_pages > 1): ?>
        <div class="bg-white rounded-lg shadow-md mt-4 px-4 py-3">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm text-gray-700">
                        Mostrando <span class="font-medium"><?php echo ($offset + 1); ?></span> a 
                        <span class="font-medium"><?php echo min($offset + $items_per_page, $total_items); ?></span> de 
                        <span class="font-medium"><?php echo $total_items; ?></span> resultados
                    </p>
                </div>
                <div>
                    <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                        <!-- Botón Primera página -->
                        <?php if ($current_page_num > 1): ?>
                        <a href="?page=stock_calculado&page_num=1&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                            <i class="fas fa-angle-double-left"></i>
                        </a>
                        <?php endif; ?>
                        
                        <!-- Botón Anterior -->
                        <?php if ($current_page_num > 1): ?>
                        <a href="?page=stock_calculado&page_num=<?php echo $current_page_num - 1; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="relative inline-flex items-center px-2 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                            <i class="fas fa-angle-left"></i>
                        </a>
                        <?php endif; ?>
                        
                        <!-- Números de página -->
                        <?php
                        $start_page = max(1, $current_page_num - 4);
                        $end_page = min($total_pages, $current_page_num + 5);
                        
                        if ($start_page > 1): ?>
                            <span class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700">...</span>
                        <?php endif;
                        
                        for ($i = $start_page; $i <= $end_page; $i++): ?>
                        <a href="?page=stock_calculado&page_num=<?php echo $i; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="<?php echo $i === $current_page_num ? 'z-10 bg-indigo-50 border-indigo-500 text-indigo-600' : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'; ?> relative inline-flex items-center px-4 py-2 border text-sm font-medium">
                            <?php echo $i; ?>
                        </a>
                        <?php endfor;
                        
                        if ($end_page < $total_pages): ?>
                            <span class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700">...</span>
                        <?php endif; ?>
                        
                        <!-- Botón Siguiente -->
                        <?php if ($current_page_num < $total_pages): ?>
                        <a href="?page=stock_calculado&page_num=<?php echo $current_page_num + 1; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="relative inline-flex items-center px-2 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                            <i class="fas fa-angle-right"></i>
                        </a>
                        <?php endif; ?>
                        
                        <!-- Botón Última página -->
                        <?php if ($current_page_num < $total_pages): ?>
                        <a href="?page=stock_calculado&page_num=<?php echo $total_pages; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page', 'page_num']))); ?>" 
                           class="relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                            <i class="fas fa-angle-double-right"></i>
                        </a>
                        <?php endif; ?>
                    </nav>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <!-- Botones de exportación -->
    <div class="bg-white p-4 rounded-lg shadow-md">
        <div class="flex gap-4">
            <form action="pages/actions/stock_calculado_export.php" method="POST" class="inline">
                <input type="hidden" name="export_mode" value="all_filtered">
                <?php foreach ($_GET as $key => $value): ?>
                    <?php if (is_array($value)): ?>
                        <?php foreach ($value as $v): ?>
                            <input type="hidden" name="<?php echo htmlspecialchars($key); ?>[]" value="<?php echo htmlspecialchars($v); ?>">
                        <?php endforeach; ?>
                    <?php else: ?>
                        <input type="hidden" name="<?php echo htmlspecialchars($key); ?>" value="<?php echo htmlspecialchars($value); ?>">
                    <?php endif; ?>
                <?php endforeach; ?>
                <button type="submit" class="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition-colors">
                    <i class="fas fa-download mr-2"></i>Exportar Filtrados
                </button>
            </form>
            
            <button onclick="exportSelected()" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors">
                <i class="fas fa-file-export mr-2"></i>Exportar Seleccionados
            </button>
        </div>
    </div>
</div>

<!-- Modal para imágenes -->
<div id="imageModal" class="hidden fixed inset-0 bg-black bg-opacity-75 z-50 flex items-center justify-center p-4">
    <div class="relative max-w-4xl max-h-full">
        <button onclick="closeImageModal()" class="absolute top-4 right-4 text-white text-2xl hover:text-gray-300">
            <i class="fas fa-times"></i>
        </button>
        <img id="modalImage" src="" alt="" class="max-w-full max-h-full object-contain">
    </div>
</div>

<!-- Modal para notas de imperfectos -->
<div id="notesModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-lg max-w-md w-full p-6">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg font-semibold">Notas del Producto</h3>
            <button onclick="closeNotesModal()" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div id="notesContent" class="text-gray-700"></div>
    </div>
</div>

<script>
// Funciones para los modales
function openImageModal(url) {
    document.getElementById('modalImage').src = url;
    document.getElementById('imageModal').classList.remove('hidden');
}

function closeImageModal() {
    document.getElementById('imageModal').classList.add('hidden');
    document.getElementById('modalImage').src = '';
}

function showNotes(notes) {
    document.getElementById('notesContent').textContent = notes;
    document.getElementById('notesModal').classList.remove('hidden');
}

function closeNotesModal() {
    document.getElementById('notesModal').classList.add('hidden');
    document.getElementById('notesContent').textContent = '';
}

// Cerrar modales con ESC
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeImageModal();
        closeNotesModal();
    }
});

// Cambiar entre vistas
function setView(view) {
    const tableView = document.getElementById('tableView');
    const cardsView = document.getElementById('cardsView');
    const viewTable = document.getElementById('viewTable');
    const viewCards = document.getElementById('viewCards');
    
    if (view === 'cards') {
        tableView.classList.add('hidden');
        cardsView.classList.remove('hidden');
        viewTable.classList.remove('bg-white', 'text-gray-700');
        viewTable.classList.add('text-gray-500');
        viewCards.classList.add('bg-white', 'text-gray-700');
        viewCards.classList.remove('text-gray-500');
        localStorage.setItem('stockView', 'cards');
    } else {
        tableView.classList.remove('hidden');
        cardsView.classList.add('hidden');
        viewTable.classList.add('bg-white', 'text-gray-700');
        viewTable.classList.remove('text-gray-500');
        viewCards.classList.remove('bg-white', 'text-gray-700');
        viewCards.classList.add('text-gray-500');
        localStorage.setItem('stockView', 'table');
    }
}

// Cargar vista guardada
document.addEventListener('DOMContentLoaded', function() {
    const savedView = localStorage.getItem('stockView') || 'table';
    setView(savedView);
    
    // Cerrar dropdowns al hacer click fuera
    document.addEventListener('click', function(e) {
        if (!e.target.closest('#bodegasDropdown') && !e.target.closest('[onclick*="bodegasDropdown"]')) {
            document.getElementById('bodegasDropdown').classList.add('hidden');
        }
        if (!e.target.closest('#categoriasDropdown') && !e.target.closest('[onclick*="categoriasDropdown"]')) {
            document.getElementById('categoriasDropdown').classList.add('hidden');
        }
    });
});

// Toggle filtros principales
function toggleFilters() {
    const content = document.getElementById('filterContent');
    const icon = document.getElementById('filterToggleIcon');
    
    if (content.classList.contains('hidden')) {
        content.classList.remove('hidden');
        icon.classList.remove('fa-chevron-down');
        icon.classList.add('fa-chevron-up');
        localStorage.setItem('filtersOpen', 'true');
    } else {
        content.classList.add('hidden');
        icon.classList.remove('fa-chevron-up');
        icon.classList.add('fa-chevron-down');
        localStorage.setItem('filtersOpen', 'false');
    }
}

// Cargar estado de filtros al iniciar
window.addEventListener('load', function() {
    const filtersOpen = localStorage.getItem('filtersOpen');
    if (filtersOpen === 'true') {
        toggleFilters();
    }
});

// Toggle dropdowns de checkboxes
function toggleDropdown(id) {
    const dropdown = document.getElementById(id);
    dropdown.classList.toggle('hidden');
}

// Toggle all checkboxes
function toggleAll(type) {
    const checkboxes = document.querySelectorAll(`.${type}-checkbox`);
    const allChecked = Array.from(checkboxes).every(cb => cb.checked);
    
    checkboxes.forEach(checkbox => {
        checkbox.checked = !allChecked;
    });
    
    updateLabel(type);
}

// Update label when checkboxes change
function updateLabel(type) {
    const checkboxes = document.querySelectorAll(`.${type}-checkbox:checked`);
    const label = document.getElementById(`${type}Label`);
    const count = checkboxes.length;
    
    if (count === 0) {
        label.textContent = type === 'bodegas' ? 'Todas las tiendas' : 'Todas las categorías';
    } else if (count === 1) {
        label.textContent = checkboxes[0].parentElement.textContent.trim();
    } else {
        label.textContent = `${count} seleccionadas`;
    }
}

// Selección de items en tabla
document.getElementById('select-all').addEventListener('change', function() {
    const checkboxes = document.querySelectorAll('.item-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.checked = this.checked;
    });
    // También sincronizar con las tarjetas
    const cardCheckboxes = document.querySelectorAll('.item-checkbox-card');
    cardCheckboxes.forEach(checkbox => {
        checkbox.checked = this.checked;
    });
});

// Sincronizar checkboxes entre vistas
document.addEventListener('change', function(e) {
    if (e.target.classList.contains('item-checkbox') || e.target.classList.contains('item-checkbox-card')) {
        const value = e.target.value;
        const isChecked = e.target.checked;
        
        // Sincronizar con el otro tipo de checkbox
        document.querySelectorAll(`input[value="${value}"]`).forEach(checkbox => {
            checkbox.checked = isChecked;
        });
    }
});

// Exportar seleccionados
function exportSelected() {
    const selected = new Set();
    
    // Obtener seleccionados de ambas vistas
    document.querySelectorAll('.item-checkbox:checked, .item-checkbox-card:checked').forEach(checkbox => {
        selected.add(checkbox.value);
    });
    
    if (selected.size === 0) {
        alert('Por favor selecciona al menos un item para exportar');
        return;
    }
    
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = 'pages/actions/stock_calculado_export.php';
    
    selected.forEach(id => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'selected_ids[]';
        input.value = id;
        form.appendChild(input);
    });
    
    document.body.appendChild(form);
    form.submit();
}
</script>