<?php
/**
 * Script para configurar el cron job de actualización automática del catálogo
 * Se ejecuta cada 6 horas: 00:00, 06:00, 12:00, 18:00
 */

date_default_timezone_set('America/Bogota');

// Configuración del cron job
$cron_entries = [
    // Actualización automática cada 6 horas
    '0 */6 * * * /usr/bin/php ' . __DIR__ . '/actualizar.php >> ' . __DIR__ . '/logs/cron.log 2>&1',
    
    // Limpieza de logs cada domingo a las 2 AM
    '0 2 * * 0 find ' . __DIR__ . '/logs -name "*.log" -mtime +30 -delete',
    
    // Verificación de salud del catálogo cada hora
    '0 * * * * curl -s -o /dev/null -w "%{http_code}" https://stock.crusardi.net/catalogo_publico/ | grep -q "200" || echo "Catálogo no responde" >> ' . __DIR__ . '/logs/health_check.log'
];

// Función para agregar entradas al crontab
function addCronEntries($entries) {
    // Obtener el crontab actual
    $current_crontab = shell_exec('crontab -l 2>/dev/null') ?: '';
    
    // Filtrar entradas existentes del catálogo
    $lines = explode("\n", $current_crontab);
    $filtered_lines = array_filter($lines, function($line) {
        return !strpos($line, 'catalogo_publico') && 
               !strpos($line, 'actualizar.php') &&
               !strpos($line, 'health_check') &&
               trim($line) !== '';
    });
    
    // Agregar nuevas entradas
    $new_crontab = implode("\n", $filtered_lines) . "\n";
    $new_crontab .= "# Entradas automáticas del catálogo público - " . date('Y-m-d H:i:s') . "\n";
    $new_crontab .= implode("\n", $entries) . "\n";
    
    // Escribir el nuevo crontab
    $temp_file = tempnam(sys_get_temp_dir(), 'crontab');
    file_put_contents($temp_file, $new_crontab);
    
    $result = shell_exec("crontab $temp_file 2>&1");
    unlink($temp_file);
    
    return $result;
}

// Función para verificar el estado del cron
function checkCronStatus() {
    $crontab = shell_exec('crontab -l 2>/dev/null') ?: '';
    $catalogo_entries = array_filter(explode("\n", $crontab), function($line) {
        return strpos($line, 'catalogo_publico') !== false || 
               strpos($line, 'actualizar.php') !== false;
    });
    
    return [
        'total_entries' => count($catalogo_entries),
        'entries' => array_values($catalogo_entries)
    ];
}

// Función para probar la actualización manual
function testUpdate() {
    $url = 'https://stock.crusardi.net/catalogo_publico/actualizar.php';
    $context = stream_context_create([
        'http' => [
            'timeout' => 60,
            'method' => 'GET'
        ]
    ]);
    
    $response = @file_get_contents($url, false, $context);
    return $response ? json_decode($response, true) : false;
}

// Procesar acción
$action = $_GET['action'] ?? 'status';

switch ($action) {
    case 'install':
        $result = addCronEntries($cron_entries);
        $status = checkCronStatus();
        
        echo json_encode([
            'success' => true,
            'message' => 'Cron job configurado exitosamente',
            'cron_result' => $result,
            'status' => $status
        ]);
        break;
        
    case 'uninstall':
        $current_crontab = shell_exec('crontab -l 2>/dev/null') ?: '';
        $lines = explode("\n", $current_crontab);
        $filtered_lines = array_filter($lines, function($line) {
            return !strpos($line, 'catalogo_publico') && 
                   !strpos($line, 'actualizar.php') &&
                   !strpos($line, 'health_check') &&
                   trim($line) !== '';
        });
        
        $new_crontab = implode("\n", $filtered_lines) . "\n";
        $temp_file = tempnam(sys_get_temp_dir(), 'crontab');
        file_put_contents($temp_file, $new_crontab);
        
        $result = shell_exec("crontab $temp_file 2>&1");
        unlink($temp_file);
        
        echo json_encode([
            'success' => true,
            'message' => 'Cron job eliminado exitosamente',
            'cron_result' => $result
        ]);
        break;
        
    case 'test':
        $test_result = testUpdate();
        echo json_encode([
            'success' => $test_result !== false,
            'test_result' => $test_result,
            'message' => $test_result ? 'Prueba exitosa' : 'Error en la prueba'
        ]);
        break;
        
    case 'status':
    default:
        $status = checkCronStatus();
        $last_update = 'No disponible';
        $last_update_file = __DIR__ . '/logs/ultima_actualizacion.txt';
        if (file_exists($last_update_file)) {
            $last_update = file_get_contents($last_update_file);
        }
        
        echo json_encode([
            'success' => true,
            'status' => $status,
            'last_update' => $last_update,
            'catalog_url' => 'https://stock.crusardi.net/catalogo_publico/'
        ]);
        break;
}
?>
