<?php
/**
 * Catálogo Público BoConcept Crusardi
 * Endpoint público que se actualiza automáticamente cada 6 horas
 */

// Configuración de base de datos
require_once '/home/crusardi/config_crusardi/database.php';

// Configurar charset UTF-8
if ($pdo) {
    $pdo->exec("SET NAMES 'utf8mb4'");
    $pdo->exec("SET CHARACTER SET utf8mb4");
    $pdo->exec("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");
}

date_default_timezone_set('America/Bogota');

// Verificar si hay actualización automática programada
if (isset($_GET['auto_update']) && $_GET['auto_update'] === 'true') {
    $auth_key = $_GET['key'] ?? '';
    $expected_key = 'crusardi_auto_update_2024';
    
    if ($auth_key !== $expected_key) {
        http_response_code(401);
        die('Unauthorized');
    }
    
    // Ejecutar actualización automática
    try {
        $stmt = $pdo->prepare("CALL consolidar_stock_crusardi()");
        $stmt->execute();
        $stmt->closeCursor();
        
        // Actualizar timestamp de última actualización
        $update_time = date('Y-m-d H:i:s');
        file_put_contents(__DIR__ . '/logs/catalogo_ultima_actualizacion.txt', $update_time);
        
        echo json_encode(['success' => true, 'updated' => $update_time]);
        exit;
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}

// Obtener datos del catálogo
try {
    // Consulta optimizada para el catálogo público
    // Agrupa por código único y suma el stock total
    // Excluye ubicación "show" y productos sin precio
    $query = "
        SELECT 
            codigo,
            codigo_unico,
            nombre,
            tipo,
            SUM(stock_total) as stock_total,
            MAX(precio) as precio,
            MAX(descuento) as descuento,
            MAX(precio_final) as precio_final,
            MAX(origen_descuento) as origen_descuento,
            MAX(categoria1) as categoria1,
            MAX(categoria2) as categoria2,
            MAX(es_imperfecto) as es_imperfecto,
            MAX(es_kit) as es_kit
        FROM stock_calculado_cache 
        WHERE stock_total > 0 
        AND precio > 0
        AND (ubicacion IS NULL OR ubicacion != 'show')
        GROUP BY codigo_unico, codigo, nombre, tipo
        ORDER BY 
            CASE 
                WHEN MAX(es_imperfecto) = 1 THEN 1
                WHEN tipo = 'kit' THEN 2
                WHEN tipo = 'articulo' THEN 3
                ELSE 4
            END,
            CASE 
                WHEN codigo REGEXP '^[0-9]+$' AND LENGTH(codigo) >= 10 THEN 1
                ELSE 2
            END,
            nombre ASC
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Obtener categorías dinámicamente desde la base de datos (solo categoria1)
    $categorias_query = "
        SELECT DISTINCT categoria1
        FROM stock_calculado_cache 
        WHERE stock_total > 0 
        AND precio > 0
        AND (ubicacion IS NULL OR ubicacion != 'show')
        AND categoria1 IS NOT NULL
        AND categoria1 != ''
        ORDER BY categoria1
    ";
    
    $categorias_stmt = $pdo->prepare($categorias_query);
    $categorias_stmt->execute();
    $categorias_data = $categorias_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Procesar categorías para el filtro (solo categoria1)
    $categorias_disponibles = [];
    foreach ($categorias_data as $cat) {
        if (!empty($cat['categoria1'])) {
            $categorias_disponibles[] = $cat['categoria1'];
        }
    }
    sort($categorias_disponibles);
    
    // Estadísticas
    $stats_query = "
        SELECT 
            COUNT(DISTINCT codigo_unico) as total_productos,
            SUM(stock_total) as total_unidades,
            SUM(CASE WHEN tipo = 'kit' THEN 1 ELSE 0 END) as total_kits,
            SUM(CASE WHEN tipo = 'articulo' THEN 1 ELSE 0 END) as total_articulos,
            SUM(CASE WHEN es_imperfecto = 1 THEN 1 ELSE 0 END) as total_imperfectos
        FROM stock_calculado_cache 
        WHERE stock_total > 0 
        AND precio > 0
        AND (ubicacion IS NULL OR ubicacion != 'show')
    ";
    
    $stats = $pdo->query($stats_query)->fetch(PDO::FETCH_ASSOC);
    
    // Leer última actualización
    $ultima_actualizacion = 'No disponible';
    $ultima_actualizacion_file = __DIR__ . '/logs/catalogo_ultima_actualizacion.txt';
    if (file_exists($ultima_actualizacion_file)) {
        $ultima_actualizacion = file_get_contents($ultima_actualizacion_file);
    }
    
} catch (Exception $e) {
    $productos = [];
    $categorias_disponibles = [];
    $stats = ['total_productos' => 0, 'total_unidades' => 0, 'total_kits' => 0, 'total_articulos' => 0, 'total_imperfectos' => 0];
    $ultima_actualizacion = 'Error al cargar datos';
}

// Función para obtener URL de imagen
function getImageUrl($codigo) {
    return "https://imagenes.crusardi.net/boconcept/STOCKCOMERCIAL/{$codigo}.jpg";
}

// Función para formatear precio
function formatPrice($price) {
    return '$' . number_format($price, 0, ',', '.');
}

// Función para determinar si es precio a consultar
function isPriceOnRequest($precio, $descuento) {
    // Solo marcar como "precio a consultar" si:
    // 1. El precio es 0 o negativo
    // 2. El precio es extremadamente alto (más de 500 millones - probablemente error)
    return $precio <= 0 || ($precio > 500000000);
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta http-equiv="Content-Type" content="text/html; charset=UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Catálogo de Productos BoConcept - Crusardi</title>
    
    <!-- SEO Meta Tags -->
    <meta name="description" content="Explora nuestra amplia gama de muebles BoConcept. Diseño danés contemporáneo, calidad excepcional. Envíos a toda Colombia.">
    <meta name="keywords" content="BoConcept, muebles, diseño danés, sofás, sillas, mesas, decoración, Crusardi">
    <meta name="robots" content="index, follow">
    
    <!-- Open Graph Meta Tags -->
    <meta property="og:title" content="Catálogo BoConcept - Crusardi">
    <meta property="og:description" content="Descubre la colección completa de muebles BoConcept. Diseño escandinavo, funcionalidad y estilo únicos.">
    <meta property="og:type" content="website">
    <meta property="og:url" content="https://stock.crusardi.net/catalogo_publico/">
    <meta property="og:image" content="https://stock.crusardi.net/assets/logo.png">
    
    <!-- Fonts and Icons -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Montserrat:wght@400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        html { scroll-behavior: smooth; }
        body { 
            font-family: 'Montserrat', 'Helvetica', 'Arial', sans-serif; 
            -webkit-font-smoothing: antialiased; 
            margin: 0; 
            padding: 0; 
            background-color: #f8f9fa; 
            color: #2c2c2c; 
        }
        
        .campaign-header { 
            background: linear-gradient(135deg, #1a1a1a 0%, #2c2c2c 100%); 
            padding: 30px; 
            text-align: center; 
            min-height: 200px; 
            display: flex; 
            flex-direction: column; 
            justify-content: center; 
            align-items: center; 
        }
        .campaign-header .title-box { 
            border: 3px solid white; 
            padding: 15px 30px; 
            display: inline-block; 
        }
        .campaign-header h1 { 
            font-size: 48px; 
            font-weight: 800; 
            margin: 0; 
            color: white; 
            text-transform: uppercase; 
            letter-spacing: 3px; 
        }
        .campaign-header h2 { 
            font-size: 18px; 
            margin: 4px 0 0 0; 
            font-weight: 600; 
            color: white; 
            text-transform: uppercase; 
            letter-spacing: 1px; 
        }
        .campaign-header .subtitle { 
            font-size: 16px; 
            color: #ffffff; 
            font-weight: 600; 
            margin-bottom: 20px; 
            text-transform: uppercase; 
        }
        
        .catalog-container { 
            width: 100%; 
            max-width: 850px; 
            margin: auto; 
            padding: 20px; 
            box-sizing: border-box; 
        }
        
        .inventory-message {
            background: linear-gradient(135deg, #2c2c2c 0%, #404040 100%);
            color: white;
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 30px;
            text-align: center;
        }
        
        .inventory-message p {
            margin: 0;
            font-size: 18px;
            font-weight: 500;
        }
        
        
        .inventory-message strong {
            color: #f4d03f;
            font-weight: 700;
        }
        
        .filter-container { 
            display: grid; 
            grid-template-columns: 1fr; 
            gap: 15px; 
            background-color: #fff; 
            padding: 20px; 
            border-radius: 12px; 
            margin-bottom: 30px; 
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08); 
            border: 1px solid #f0f0f0;
        }
        @media (min-width: 768px) { 
            .filter-container { 
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); 
            } 
        }
        .filter-select { 
            width: 100%; 
            padding: 10px 15px; 
            border: 1px solid #e0e0e0; 
            border-radius: 8px; 
            font-family: 'Montserrat', sans-serif; 
            font-size: 14px; 
            background-color: #fff; 
            transition: border-color 0.3s ease;
        }
        .filter-select:focus {
            outline: none;
            border-color: #f4d03f;
            box-shadow: 0 0 0 2px rgba(244, 208, 63, 0.2);
        }
        .filter-container label { 
            display: block; 
            font-weight: 600; 
            font-size: 12px; 
            margin-bottom: 5px; 
            color: #2c2c2c; 
        }
        
        .products-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 25px;
        }
        
        .product-card { 
            position: relative;
            display: flex; 
            background-color: #ffffff; 
            border-radius: 12px; 
            overflow: hidden; 
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08); 
            border: 1px solid #f0f0f0;
            transition: transform 0.2s ease-in-out, box-shadow 0.2s ease-in-out;
        }
        
        .product-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
        }

        .product-image-wrapper { 
            display: block; 
            text-decoration: none; 
        }
        .product-image-container { 
            position: relative; 
            width: 280px; 
            height: 280px; 
            flex-shrink: 0; 
            background-color: #fff; 
        }
        .product-image-container img { 
            width: 100%; 
            height: 100%; 
            object-fit: contain; 
            display: block; 
        }
        .discount-badge { 
            position: absolute; 
            top: 15px; 
            left: 15px; 
            background-color: #e74c3c; 
            color: #fff; 
            font-size: 14px; 
            font-family: 'Montserrat', sans-serif; 
            font-weight: 700; 
            width: 55px; 
            height: 55px; 
            border-radius: 50%; 
            border: 2px solid #fff; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            text-align: center; 
            line-height: 1; 
        }
        .status-tag { 
            position: absolute; 
            bottom: 10px; 
            left: 50%; 
            transform: translateX(-50%); 
            background-color: rgba(44, 44, 44, 0.9); 
            color: #fff; 
            font-size: 14px; 
            font-family: 'Montserrat', sans-serif; 
            font-weight: 700; 
            padding: 6px 12px; 
            border-radius: 4px; 
            text-transform: uppercase; 
        }
        .product-info { 
            display: flex; 
            flex-direction: column; 
            padding: 25px; 
            flex-grow: 1; 
        }
        .product-details { 
            flex-grow: 1; 
        }
        .product-info .name { 
            font-size: 20px; 
            font-weight: 700; 
            margin: 0 0 5px 0; 
            line-height: 1.3; 
            text-transform: uppercase; 
            color: #2c2c2c; 
        }
        .product-info .ref { 
            font-size: 13px; 
            color: #666666; 
            margin-bottom: 10px; 
            font-weight: 500; 
        }
        .product-info .category { 
            font-size: 12px; 
            color: #2c2c2c; 
            margin-bottom: 15px; 
            font-weight: 600; 
            text-transform: uppercase; 
            background-color: #f4d03f; 
            padding: 4px 8px; 
            border-radius: 4px; 
            display: inline-block; 
        }
        .price-and-action { 
            display: flex; 
            flex-direction: column; 
            align-items: flex-start; 
            gap: 15px; 
            margin-top: auto; 
            padding-top: 20px; 
            border-top: 1px solid #f0f0f0; 
        }
        .price-item { 
            margin-bottom: 5px; 
        }
        .price-label { 
            font-weight: 600; 
            font-size: 12px; 
            color: #666666; 
            display: block; 
            text-transform: uppercase; 
            margin-bottom: 2px; 
        }
        .price-old { 
            text-decoration: line-through; 
            color: #999999; 
            font-size: 18px; 
            font-weight: 500; 
        }
        .price-new { 
            font-size: 28px; 
            font-weight: 800; 
            color: #2c2c2c; 
        }
        .price-new.sale { 
            color: #e74c3c; 
        }
        
        .discount-pill {
            display: inline-block;
            background-color: #e74c3c;
            color: #fff;
            font-weight: 800;
            font-size: 12px;
            padding: 3px 8px;
            border-radius: 6px;
            margin-left: 8px;
            line-height: 1;
        }
        
        .price-consult { 
            font-size: 24px; 
            font-weight: 800; 
            color: #f4d03f; 
            text-transform: uppercase;
            letter-spacing: 1px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.7; }
        }
        
        .product-card.price-on-request {
            border: 2px solid #f4d03f;
            box-shadow: 0 8px 25px rgba(244, 208, 63, 0.25);
        }
        
        .status-tag.status-consult {
            background-color: rgba(244, 208, 63, 0.95);
            color: #2c2c2c;
            animation: pulse 2s infinite;
        }
        
        .contact-button {
            background-color: #2c2c2c;
            color: white;
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            font-family: 'Montserrat', sans-serif;
            font-weight: 600;
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-block;
            text-align: center;
        }
        
        .contact-button:hover {
            background-color: #1a1a1a;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(44, 44, 44, 0.4);
        }
        
        .contact-button-special {
            background-color: #f4d03f !important;
            color: #2c2c2c !important;
            animation: subtle-glow 2s infinite;
        }
        
        .contact-button-special:hover {
            background-color: #f1c40f !important;
            color: #2c2c2c !important;
        }
        
        @keyframes subtle-glow {
            0%, 100% { box-shadow: 0 4px 12px rgba(244, 208, 63, 0.4); }
            50% { box-shadow: 0 4px 20px rgba(244, 208, 63, 0.6); }
        }
        
        .quantity-tag { 
            font-size: 14px; 
            font-weight: 700; 
            color: #2c2c2c; 
            background-color: #f8f9fa; 
            border: 1px solid #e9ecef;
            padding: 6px 12px; 
            border-radius: 6px; 
            display: inline-block; 
            margin-top: 10px; 
        }
        
        .loading {
            text-align: center;
            padding: 40px;
            font-size: 18px;
            color: #2c2c2c;
        }
        
        .error-message {
            background-color: #fef2f2;
            border: 1px solid #fecaca;
            color: #dc2626;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
            margin: 20px;
        }
        
        .last-updated {
            text-align: center;
            color: #666666;
            font-size: 12px;
            margin-bottom: 20px;
        }
        
        .broken-image-product {
            opacity: 0.8;
            order: 999;
        }
        
        .broken-image-product .product-image-container::after {
            content: "Sin imagen";
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0, 0, 0, 0.7);
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 600;
            z-index: 10;
            pointer-events: none;
        }
        
        @media (max-width: 768px) {
            .campaign-header h1 { font-size: 32px; }
            .campaign-header h2 { font-size: 14px; }
            .product-card { flex-direction: column; }
            .product-image-container { width: 100%; height: 200px; }
            .products-grid { grid-template-columns: 1fr; }
            .inventory-message p { font-size: 16px; }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="campaign-header">
    <div class="subtitle">Catálogo de Productos</div>
        <div class="title-box">
            <h1>BoConcept</h1>
            <h2>Entrega Inmediata</h2>
        </div>
        
    </div>

    <div class="catalog-container">
        <!-- Mensaje de inventario -->
        <div class="inventory-message">
            <p>Tenemos más de <strong><?php echo number_format(round($stats['total_unidades'], -2)); ?></strong> unidades disponibles para ti</p>
        </div>

        <!-- Filtros -->
        <div class="filter-container">
            <div>
                <label for="sort-products">Ordenar por</label>
                <select id="sort-products" class="filter-select" onchange="sortProducts()">
                    <option value="">Orden por defecto</option>
                    <option value="price-asc">Precio: menor a mayor</option>
                    <option value="price-desc">Precio: mayor a menor</option>
                    <option value="discount-desc">Mayor descuento</option>
                    <option value="name-asc">Nombre: A-Z</option>
                    <option value="stock-desc">Más unidades disponibles</option>
                </select>
            </div>
            <div>
                <label for="filter-category">Categoría</label>
                <select id="filter-category" class="filter-select" onchange="filterProducts()">
                    <option value="">Todas las categorías</option>
                    <?php foreach ($categorias_disponibles as $categoria): ?>
                        <option value="<?php echo strtolower($categoria); ?>"><?php echo htmlspecialchars($categoria); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label for="search">Buscar producto</label>
                <input type="text" id="search" class="filter-select" placeholder="Ej: lampara, lamparas, sofa, mesas..." onkeyup="filterProducts()">
            </div>
        </div>

        <!-- Última actualización -->
        <div class="last-updated">
            <i class="fas fa-clock"></i> Última actualización: <?php echo $ultima_actualizacion; ?>
        </div>

        <!-- Lista de productos -->
        <div id="products-container" class="products-grid">
            <?php if (empty($productos)): ?>
                <div class="error-message">
                    <i class="fas fa-exclamation-triangle"></i> No hay productos disponibles en este momento.
                </div>
            <?php else: ?>
                <?php foreach ($productos as $producto): ?>
                    <?php 
                    $isPriceOnRequest = isPriceOnRequest($producto['precio'], $producto['descuento']);
                    $imageUrl = getImageUrl($producto['codigo']);
                    $discountPercent = $producto['descuento'] > 0 ? round($producto['descuento']) : 0;
                    $productType = '';
                    
                    if ($producto['es_imperfecto']) {
                        $productType = 'imperfecto';
                    } elseif ($producto['es_kit']) {
                        $productType = 'kit';
                    } elseif ($producto['tipo'] === 'articulo') {
                        $productType = 'articulo';
                    }
                    ?>
                    <div class="product-card <?php echo $isPriceOnRequest ? 'price-on-request' : ''; ?>" 
                         data-type="<?php echo $productType; ?>"
                         data-price="<?php echo $producto['precio']; ?>"
                         data-discount="<?php echo $discountPercent; ?>"
                         data-stock="<?php echo $producto['stock_total']; ?>"
                         data-category="<?php echo strtolower($producto['categoria1'] ?? ''); ?>"
                         data-name="<?php echo strtolower($producto['nombre']); ?>"
                         data-code="<?php echo strtolower($producto['codigo']); ?>">
                        
                        <a href="<?php echo $imageUrl; ?>" target="_blank" class="product-image-wrapper">
                            <div class="product-image-container">
                                <img src="<?php echo $imageUrl; ?>" 
                                     alt="<?php echo htmlspecialchars($producto['nombre']); ?>"
                                     onerror="handleImageError(this)"
                                     onload="handleImageLoad(this)">
                                
                                <?php if ($discountPercent > 0 && !$isPriceOnRequest): ?>
                                    <div class="discount-badge">
                                        -<?php echo $discountPercent; ?>%
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($isPriceOnRequest): ?>
                                    <div class="status-tag status-consult">
                                        Precio a consultar
                                    </div>
                                <?php endif; ?>
                            </div>
                        </a>
                        
                        <div class="product-info">
                            <div class="product-details">
                                <h3 class="name"><?php echo htmlspecialchars($producto['nombre']); ?></h3>
                                <p class="ref">Ref: <?php echo htmlspecialchars($producto['codigo']); ?></p>
                                
                                <?php if ($producto['codigo_unico']): ?>
                                    <p class="ref">Código único: <?php echo htmlspecialchars($producto['codigo_unico']); ?></p>
                                <?php endif; ?>
                                
                                <?php if ($producto['categoria1']): ?>
                                    <div class="category"><?php echo htmlspecialchars($producto['categoria1']); ?></div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="price-and-action">
                                <div class="price-and-qty">
                                    <?php if ($isPriceOnRequest): ?>
                                        <div class="price-consult">Precio a consultar</div>
                                    <?php else: ?>
                                        <?php if ($producto['descuento'] > 0): ?>
                                            <div class="price-item">
                                                <span class="price-label">Precio anterior</span>
                                                <span class="price-old"><?php echo formatPrice($producto['precio']); ?></span>
                                            </div>
                                            <div class="price-item">
                                                <span class="price-label">Precio especial</span>
                                                <span class="price-new sale"><?php echo formatPrice($producto['precio_final']); ?></span>
                                                <span class="discount-pill">-<?php echo round($producto['descuento']); ?>%</span>
                                            </div>
                                        <?php else: ?>
                                            <div class="price-item">
                                                <span class="price-label">Precio</span>
                                                <span class="price-new"><?php echo formatPrice($producto['precio']); ?></span>
                                            </div>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                    
                                    <div class="quantity-tag">
                                        <i class="fas fa-box"></i> <?php echo number_format($producto['stock_total']); ?> unidades
                                    </div>
                                </div>
                                
                                <a href="https://wa.me/573202696125?text=Hola, me interesa el producto <?php echo urlencode($producto['nombre'] . ' - Ref: ' . $producto['codigo']); ?>" 
                                   target="_blank"
                                   class="contact-button <?php echo $isPriceOnRequest ? 'contact-button-special' : ''; ?>">
                                    <?php if ($isPriceOnRequest): ?>
                                        <i class="fas fa-phone"></i> Consultar precio
                                    <?php else: ?>
                                        <i class="fab fa-whatsapp"></i> Más información
                                    <?php endif; ?>
                                </a>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Variables globales para manejar imágenes
        let imagesLoaded = 0;
        let totalImages = 0;
        let brokenImages = new Set();
        
        // Función para normalizar texto (quitar tildes y acentos)
        function normalizeText(text) {
            return text
                .toLowerCase()
                .normalize("NFD")
                .replace(/[\u0300-\u036f]/g, "") // Quitar diacríticos (tildes, acentos)
                .replace(/[^a-z0-9\s]/g, '') // Quitar caracteres especiales excepto espacios
                .trim();
        }
        
        // Función para generar variaciones de una palabra (singular/plural)
        function generateWordVariations(word) {
            const variations = new Set();
            variations.add(word);
            
            // Reglas básicas de plurales en español
            if (word.endsWith('s')) {
                // Si termina en 's', quitar la 's' (plural -> singular)
                variations.add(word.slice(0, -1));
                // También probar quitando 'es' si termina en 'es'
                if (word.endsWith('es') && word.length > 2) {
                    variations.add(word.slice(0, -2));
                }
            } else {
                // Si no termina en 's', agregar 's' (singular -> plural)
                variations.add(word + 's');
                // También agregar 'es' para palabras que terminan en consonante
                const lastChar = word.charAt(word.length - 1);
                if ('bcdfghjklmnpqrstvwxyz'.includes(lastChar)) {
                    variations.add(word + 'es');
                }
            }
            
            // Casos especiales comunes
            const specialCases = {
                'lampara': ['lamparas'],
                'lamparas': ['lampara'],
                'sofa': ['sofas'],
                'sofas': ['sofa'],
                'mesa': ['mesas'],
                'mesas': ['mesa'],
                'silla': ['sillas'],
                'sillas': ['silla'],
                'cama': ['camas'],
                'camas': ['cama'],
                'armario': ['armarios'],
                'armarios': ['armario'],
                'estante': ['estantes'],
                'estantes': ['estante'],
                'escritorio': ['escritorios'],
                'escritorios': ['escritorio']
            };
            
            if (specialCases[word]) {
                specialCases[word].forEach(variant => variations.add(variant));
            }
            
            return Array.from(variations);
        }
        
        // Función para manejar errores de imagen
        function handleImageError(img) {
            img.src = 'https://imagenes.crusardi.net/boconcept/sin_foto.png';
            const productCard = img.closest('.product-card');
            if (productCard) {
                productCard.setAttribute('data-image-status', 'broken');
                brokenImages.add(productCard);
            }
            checkAllImagesLoaded();
        }
        
        // Función para manejar imágenes cargadas correctamente
        function handleImageLoad(img) {
            const productCard = img.closest('.product-card');
            if (productCard && img.src.includes('sin_foto.png')) {
                productCard.setAttribute('data-image-status', 'broken');
                brokenImages.add(productCard);
            } else if (productCard) {
                productCard.setAttribute('data-image-status', 'loaded');
            }
            checkAllImagesLoaded();
        }
        
        // Verificar si todas las imágenes han sido procesadas
        function checkAllImagesLoaded() {
            imagesLoaded++;
            if (imagesLoaded >= totalImages) {
                setTimeout(reorganizeProducts, 500); // Pequeño delay para asegurar que todo esté listo
            }
        }
        
        // Reorganizar productos moviendo los de imagen rota al final
        function reorganizeProducts() {
            const container = document.getElementById('products-container');
            const allCards = Array.from(container.querySelectorAll('.product-card'));
            
            // Separar productos con y sin imágenes
            const goodImages = allCards.filter(card => card.getAttribute('data-image-status') !== 'broken');
            const brokenImageCards = allCards.filter(card => card.getAttribute('data-image-status') === 'broken');
            
            // Limpiar contenedor
            container.innerHTML = '';
            
            // Agregar primero los productos con buenas imágenes
            goodImages.forEach(card => container.appendChild(card));
            
            // Agregar al final los productos con imágenes rotas
            brokenImageCards.forEach(card => {
                // Agregar una clase especial para identificar visualmente
                card.classList.add('broken-image-product');
                container.appendChild(card);
            });
            
            console.log(`Reorganizado: ${goodImages.length} productos con imágenes válidas, ${brokenImageCards.length} con imágenes rotas`);
        }
        
        function sortProducts() {
            const sortValue = document.getElementById('sort-products').value;
            const container = document.getElementById('products-container');
            const cards = Array.from(container.querySelectorAll('.product-card'));
            
            if (!sortValue) {
                // Orden por defecto - reorganizar según imágenes
                reorganizeProducts();
                return;
            }
            
            cards.sort((a, b) => {
                switch (sortValue) {
                    case 'price-asc':
                        const priceA = parseFloat(a.dataset.price) || 0;
                        const priceB = parseFloat(b.dataset.price) || 0;
                        return priceA - priceB;
                        
                    case 'price-desc':
                        const priceDescA = parseFloat(a.dataset.price) || 0;
                        const priceDescB = parseFloat(b.dataset.price) || 0;
                        return priceDescB - priceDescA;
                        
                    case 'discount-desc':
                        const discountA = parseFloat(a.dataset.discount) || 0;
                        const discountB = parseFloat(b.dataset.discount) || 0;
                        return discountB - discountA;
                        
                    case 'name-asc':
                        const nameA = a.dataset.name || '';
                        const nameB = b.dataset.name || '';
                        return nameA.localeCompare(nameB);
                        
                    case 'stock-desc':
                        const stockA = parseInt(a.dataset.stock) || 0;
                        const stockB = parseInt(b.dataset.stock) || 0;
                        return stockB - stockA;
                        
                    default:
                        return 0;
                }
            });
            
            // Limpiar y reagregar en el nuevo orden
            container.innerHTML = '';
            cards.forEach(card => container.appendChild(card));
            
            console.log(`Productos ordenados por: ${sortValue}`);
        }
        
        function filterProducts() {
            const categoryFilter = document.getElementById('filter-category').value;
            const searchFilter = normalizeText(document.getElementById('search').value);
            
            const productCards = document.querySelectorAll('.product-card');
            
            productCards.forEach(card => {
                let show = true;
                
                // Filtro por categoría
                if (categoryFilter && !normalizeText(card.dataset.category).includes(normalizeText(categoryFilter))) {
                    show = false;
                }
                
                // Filtro por búsqueda inteligente (sin tildes/acentos + plurales/singulares)
                if (searchFilter) {
                    const productName = normalizeText(card.dataset.name);
                    const productCode = normalizeText(card.dataset.code);
                    
                    // Buscar cada palabra del filtro en el nombre o código
                    const searchWords = searchFilter.split(' ').filter(word => word.length > 0);
                    const foundAllWords = searchWords.every(searchWord => {
                        // Generar variaciones de la palabra de búsqueda
                        const wordVariations = generateWordVariations(searchWord);
                        
                        // Verificar si alguna variación coincide
                        return wordVariations.some(variation => 
                            productName.includes(variation) || productCode.includes(variation)
                        );
                    });
                    
                    if (!foundAllWords) {
                        show = false;
                    }
                }
                
                card.style.display = show ? 'flex' : 'none';
            });
            
            // Actualizar contador de productos visibles
            const visibleProducts = document.querySelectorAll('.product-card[style*="flex"]').length;
            console.log(`Mostrando ${visibleProducts} productos`);
            
            // Aplicar ordenamiento después del filtrado
            sortProducts();
        }
        
        // Filtro inicial
        document.addEventListener('DOMContentLoaded', function() {
            // Contar total de imágenes
            totalImages = document.querySelectorAll('.product-image-container img').length;
            console.log(`Total de imágenes a cargar: ${totalImages}`);
            
            // Si no hay imágenes, aplicar filtros directamente
            if (totalImages === 0) {
                filterProducts();
            }
        });
    </script>
</body>
</html>