<?php
// pages/importar_productos.php

$error_message = null;
$success_message = null;
$current_role = $_SESSION['user_role'];
$log_messages = [];

// URL base para las imágenes
define('IMAGE_BASE_URL', 'https://imagenes.crusardi.net/boconcept/STOCKCOMERCIAL/');

// ELIMINAR COLUMNAS DE CATEGORÍAS SI EXISTEN (ejecutar una sola vez)
try {
    // Verificar si existen las columnas de categorías y eliminarlas
    $check_col1 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'categoria1'");
    if ($check_col1->rowCount() > 0) {
        $pdo->exec("ALTER TABLE references_data DROP COLUMN categoria1");
    }
    
    $check_col2 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'categoria2'");
    if ($check_col2->rowCount() > 0) {
        $pdo->exec("ALTER TABLE references_data DROP COLUMN categoria2");
    }
    
    // También eliminar la columna antigua 'categoria' si existe
    $check_old = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'categoria'");
    if ($check_old->rowCount() > 0) {
        $pdo->exec("ALTER TABLE references_data DROP COLUMN categoria");
    }
} catch (PDOException $e) {
    // Ignorar errores si las columnas no existen
}

// Verificar y crear otras columnas necesarias
try {
    // Verificar si existe tipo_articulo
    $check_col3 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'tipo_articulo'");
    if ($check_col3->rowCount() == 0) {
        $pdo->exec("ALTER TABLE references_data ADD COLUMN tipo_articulo VARCHAR(50)");
    }
    
    // Verificar si existe grupo_articulo
    $check_col4 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'grupo_articulo'");
    if ($check_col4->rowCount() == 0) {
        $pdo->exec("ALTER TABLE references_data ADD COLUMN grupo_articulo VARCHAR(100)");
    }
    
    // Verificar si existe estado
    $check_col5 = $pdo->query("SHOW COLUMNS FROM references_data LIKE 'estado'");
    if ($check_col5->rowCount() == 0) {
        $pdo->exec("ALTER TABLE references_data ADD COLUMN estado VARCHAR(50)");
    }
} catch (PDOException $e) {
    // Ignorar errores si las columnas ya existen
}

// Funciones auxiliares
if (!function_exists('normalize_header')) {
    function normalize_header($str) {
        $str = strtolower(trim($str));
        $unwanted_array = ['á' => 'a', 'é' => 'e', 'í' => 'i', 'ó' => 'o', 'ú' => 'u', ' ' => '_', "\t" => '_', "\n" => '_', "\r" => '_'];
        $str = strtr($str, $unwanted_array);
        return preg_replace('/_+/', '_', $str);
    }
}

// Función para obtener categorías de un producto basándose SOLO en su grupo
if (!function_exists('obtener_categorias_producto')) {
    function obtener_categorias_producto($pdo, $grupo_articulo) {
        // Si no hay grupo, retornar "Sin definir"
        if (empty($grupo_articulo) || trim($grupo_articulo) === '') {
            return ['cat1' => 'Sin definir', 'cat2' => 'Sin definir'];
        }
        
        $grupo_upper = strtoupper(trim($grupo_articulo));
        
        // Buscar en la tabla de mapeo - coincidencia exacta
        $stmt = $pdo->prepare("
            SELECT 
                COALESCE(c1.nombre, 'Sin definir') as cat1, 
                COALESCE(c2.nombre, 'Sin definir') as cat2
            FROM categoria_mapping cm
            LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
            LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
            WHERE cm.grupo_articulo = ? AND cm.activo = 1
        ");
        $stmt->execute([$grupo_upper]);
        $result = $stmt->fetch();
        
        if ($result) {
            return ['cat1' => $result['cat1'], 'cat2' => $result['cat2']];
        }
        
        // Si no hay coincidencia exacta, buscar coincidencia parcial
        $stmt = $pdo->prepare("
            SELECT 
                COALESCE(c1.nombre, 'Sin definir') as cat1, 
                COALESCE(c2.nombre, 'Sin definir') as cat2
            FROM categoria_mapping cm
            LEFT JOIN categorias_nivel1 c1 ON cm.categoria1_id = c1.id
            LEFT JOIN categorias_nivel2 c2 ON cm.categoria2_id = c2.id
            WHERE ? LIKE CONCAT('%', cm.grupo_articulo, '%') AND cm.activo = 1 
            ORDER BY LENGTH(cm.grupo_articulo) DESC 
            LIMIT 1
        ");
        $stmt->execute([$grupo_upper]);
        $result = $stmt->fetch();
        
        if ($result) {
            return ['cat1' => $result['cat1'], 'cat2' => $result['cat2']];
        }
        
        // Si no se encuentra mapeo, retornar "Sin definir"
        return ['cat1' => 'Sin definir', 'cat2' => 'Sin definir'];
    }
}

// --- LÓGICA DE PROCESAMIENTO DE ACCIONES (POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && in_array($current_role, ['admin', 'editor'])) {
    $action = $_POST['action'] ?? '';

    try {
        if ($action === 'import_paste' && !empty($_POST['paste_data'])) {
            $startTime = microtime(true);
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Proceso de importación iniciado.";
            
            $pdo->beginTransaction();
            
            $lines = explode("\n", trim($_POST['paste_data']));
            if (count($lines) < 2) {
                throw new Exception("No se encontraron datos válidos para importar");
            }
            
            // Procesar encabezados
            $headers = str_getcsv($lines[0], "\t");
            $delimiter = "\t";
            
            if (count($headers) < 2) {
                $headers = str_getcsv($lines[0], ";");
                $delimiter = ";";
            }
            
            if (count($headers) < 2) {
                $headers = str_getcsv($lines[0], ",");
                $delimiter = ",";
            }
            
            // Normalizar encabezados
            $normalized_headers = array_map('normalize_header', $headers);
            
            // ============ SECCIÓN CORREGIDA - DETECCIÓN DE ÍNDICES ============
            // Asignación directa basada en el formato conocido del archivo
            $codigo_index = false;
            $nombre_index = false;
            $grupo_index = false;
            $tipo_index = false;
            $estado_index = false;
            $precio_index = false;
            
            // Buscar columnas por posición Y contenido para mayor precisión
            foreach ($normalized_headers as $index => $header) {
                // Código de artículo - Primera columna
                if ($index == 0 && strpos($header, 'codigo') !== false) {
                    $codigo_index = 0;
                }
                
                // Nombre del artículo - Tercera columna  
                if ($index == 2 && strpos($header, 'nombre') !== false && strpos($header, 'articulo') !== false) {
                    $nombre_index = 2;
                }
                
                // Estado - Cuarta columna
                if ($index == 3 && $header === 'estado') {
                    $estado_index = 3;
                }
                
                // Grupo de artículos - Quinta columna
                if ($index == 4 && strpos($header, 'grupo') !== false && strpos($header, 'articulo') !== false) {
                    $grupo_index = 4;
                }
                
                // Tipo de artículo - Sexta columna
                if ($index == 5 && strpos($header, 'tipo') !== false && strpos($header, 'articulo') !== false) {
                    $tipo_index = 5;
                }
                
                // Precio - Séptima columna
                if ($index == 6 && $header === 'precio') {
                    $precio_index = 6;
                }
            }
            
            // Fallback: Si la detección automática falla, usar posiciones fijas conocidas
            if ($codigo_index === false) $codigo_index = 0;
            if ($nombre_index === false) $nombre_index = 2;
            if ($estado_index === false) $estado_index = 3;
            if ($grupo_index === false) $grupo_index = 4;
            if ($tipo_index === false) $tipo_index = 5;
            if ($precio_index === false) $precio_index = 6;
            
            // Log de debugging para ver qué índices se detectaron
            $log_messages[] = "[DEBUG] Índices detectados:";
            $log_messages[] = "  - Código (col 0): " . $codigo_index;
            $log_messages[] = "  - Nombre (col 2): " . $nombre_index;
            $log_messages[] = "  - Estado (col 3): " . $estado_index;
            $log_messages[] = "  - Grupo (col 4): " . $grupo_index;
            $log_messages[] = "  - Tipo (col 5): " . $tipo_index;
            $log_messages[] = "  - Precio (col 6): " . $precio_index;
            // ============ FIN SECCIÓN CORREGIDA ============
            
            if ($codigo_index === false) {
                throw new Exception("No se encontró la columna de código/referencia");
            }
            
            // Preparar statements
            $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM references_data WHERE referencia = ?");
            $insert_stmt = $pdo->prepare("
                INSERT INTO references_data 
                (referencia, nombre, tipo_articulo, grupo_articulo, estado, url_imagen, precio, precio_final) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $update_stmt = $pdo->prepare("
                UPDATE references_data SET 
                nombre = ?, tipo_articulo = ?, grupo_articulo = ?, estado = ?, precio = ?, precio_final = ? 
                WHERE referencia = ?
            ");
            
            $imported = 0;
            $updated = 0;
            $errors = 0;
            $sin_grupo_count = 0;
            
            // Procesar líneas
            for ($i = 1; $i < count($lines); $i++) {
                if (empty(trim($lines[$i]))) continue;
                
                $data = str_getcsv($lines[$i], $delimiter);
                
                // Validar que hay suficientes columnas
                if (count($data) <= $codigo_index) {
                    $log_messages[] = "[WARN] " . date('H:i:s') . " - Línea $i omitida: datos incompletos";
                    $errors++;
                    continue;
                }
                
                try {
                    $codigo = trim($data[$codigo_index] ?? '');
                    if (empty($codigo)) {
                        $log_messages[] = "[WARN] " . date('H:i:s') . " - Línea $i omitida: sin código";
                        $errors++;
                        continue;
                    }
                    
                    // Extraer datos con los índices correctos
                    $nombre = isset($data[$nombre_index]) ? trim($data[$nombre_index]) : '';
                    $grupo_articulo = isset($data[$grupo_index]) ? trim($data[$grupo_index]) : '';
                    $tipo_articulo = isset($data[$tipo_index]) ? trim($data[$tipo_index]) : 'Artículo';
                    $estado = isset($data[$estado_index]) ? trim($data[$estado_index]) : '';
                    
                    // Log para debugging de productos específicos
                    if ($codigo == '104011010520' || $codigo == '104011010510') {
                        $log_messages[] = "[DEBUG] Producto $codigo:";
                        $log_messages[] = "  - Nombre: '$nombre'";
                        $log_messages[] = "  - Estado: '$estado'";
                        $log_messages[] = "  - Grupo: '$grupo_articulo'";
                        $log_messages[] = "  - Tipo: '$tipo_articulo'";
                    }
                    
                    // Procesar precio
                    $precio = 0;
                    if (isset($data[$precio_index])) {
                        $precio_str = $data[$precio_index];
                        $precio_str = str_replace('.', '', $precio_str); // Quitar puntos de miles
                        $precio_str = str_replace(',', '.', $precio_str); // Cambiar coma por punto decimal
                        $precio = floatval($precio_str);
                    }
                    
                    // Contar productos sin grupo (solo para logging)
                    if (empty($grupo_articulo)) {
                        $sin_grupo_count++;
                        $log_messages[] = "[INFO] " . date('H:i:s') . " - Producto '$codigo' sin grupo definido";
                    }
                    
                    // URL de imagen
                    $url_imagen = IMAGE_BASE_URL . $codigo . '.jpg';
                    
                    // Verificar si existe
                    $check_stmt->execute([$codigo]);
                    $exists = $check_stmt->fetchColumn() > 0;
                    
                    if ($exists) {
                        // ACTUALIZAR producto existente
                        $update_stmt->execute([$nombre, $tipo_articulo, $grupo_articulo, $estado, $precio, $precio, $codigo]);
                        $updated++;
                        if ($codigo == '104011010520') {
                            $log_messages[] = "[INFO] Producto $codigo ACTUALIZADO con estado='$estado', grupo='$grupo_articulo'";
                        }
                    } else {
                        // INSERTAR nuevo producto
                        $insert_stmt->execute([$codigo, $nombre, $tipo_articulo, $grupo_articulo, $estado, $url_imagen, $precio, $precio]);
                        $imported++;
                    }
                    
                } catch (PDOException $e) {
                    $log_messages[] = "[ERROR] " . date('H:i:s') . " - Error en línea $i: " . $e->getMessage();
                    $errors++;
                }
            }
            
            $pdo->commit();
            
            $duration = round(microtime(true) - $startTime, 2);
            $success_message = "Importación completada en {$duration}s. Importados: $imported, Actualizados: $updated, Errores: $errors";
            
            if ($sin_grupo_count > 0) {
                $success_message .= ". Productos sin grupo: $sin_grupo_count (tendrán categoría 'Sin definir')";
            }
            
            $log_messages[] = "[SUCCESS] " . date('H:i:s') . " - " . $success_message;
            
        } elseif ($action === 'create' || $action === 'edit') {
            // Crear o editar producto individual
            $referencia = trim($_POST['referencia'] ?? '');
            $nombre = trim($_POST['nombre'] ?? '');
            $grupo_articulo = trim($_POST['grupo_articulo'] ?? '');
            $tipo_articulo = trim($_POST['tipo_articulo'] ?? 'Artículo');
            $estado = trim($_POST['estado'] ?? '');
            $precio = floatval($_POST['precio'] ?? 0);
            
            if (empty($referencia)) {
                throw new Exception("La referencia es obligatoria");
            }
            
            $url_imagen = IMAGE_BASE_URL . $referencia . '.jpg';
            
            if ($action === 'create') {
                // Verificar si la referencia ya existe
                $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM references_data WHERE referencia = ?");
                $check_stmt->execute([$referencia]);
                
                if ($check_stmt->fetchColumn() > 0) {
                    throw new Exception("La referencia '$referencia' ya existe. Use la opción de editar para modificarla.");
                }
                
                $stmt = $pdo->prepare("INSERT INTO references_data (referencia, nombre, tipo_articulo, grupo_articulo, estado, url_imagen, precio, precio_final) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$referencia, $nombre, $tipo_articulo, $grupo_articulo, $estado, $url_imagen, $precio, $precio]);
                $success_message = "Producto creado correctamente";
            } else {
                // Verificar que el producto existe antes de actualizar
                $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM references_data WHERE referencia = ?");
                $check_stmt->execute([$referencia]);
                
                if ($check_stmt->fetchColumn() == 0) {
                    throw new Exception("La referencia '$referencia' no existe.");
                }
                
                $stmt = $pdo->prepare("UPDATE references_data SET nombre = ?, tipo_articulo = ?, grupo_articulo = ?, estado = ?, precio = ?, precio_final = ? WHERE referencia = ?");
                $stmt->execute([$nombre, $tipo_articulo, $grupo_articulo, $estado, $precio, $precio, $referencia]);
                $success_message = "Producto actualizado correctamente";
            }
            
        } elseif ($action === 'delete') {
            $referencia = trim($_POST['referencia'] ?? '');
            if (empty($referencia)) {
                throw new Exception("La referencia es obligatoria para eliminar");
            }
            
            // Verificar que el producto existe antes de eliminar
            $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM references_data WHERE referencia = ?");
            $check_stmt->execute([$referencia]);
            
            if ($check_stmt->fetchColumn() == 0) {
                throw new Exception("La referencia '$referencia' no existe.");
            }
            
            $stmt = $pdo->prepare("DELETE FROM references_data WHERE referencia = ?");
            $stmt->execute([$referencia]);
            $success_message = "Producto '$referencia' eliminado correctamente";
        }
        
    } catch (PDOException $e) {
        // Solo hacer rollback si hay una transacción activa
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        // Mejorar mensaje de error para duplicados
        if ($e->getCode() == '23000') {
            $error_message = "Error: La referencia ya existe o hay un problema de integridad de datos.";
        } else {
            $error_message = "Error de base de datos: " . $e->getMessage();
        }
        $log_messages[] = "[ERROR] " . date('H:i:s') . " - " . $error_message;
    } catch (Exception $e) {
        // Solo hacer rollback si hay una transacción activa
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $error_message = $e->getMessage();
        $log_messages[] = "[ERROR] " . date('H:i:s') . " - " . $error_message;
    }
}

// --- CONSULTA DE PRODUCTOS ---
$search_term = $_GET['search'] ?? '';
$filter_grupo = $_GET['grupo'] ?? '';
$filter_cat1 = $_GET['cat1'] ?? '';
$current_page_num = max(1, intval($_GET['page_num'] ?? 1));
$items_per_page = 50;

// Primero obtener TODOS los productos para poder filtrar por categoría (que es dinámica)
$query = "SELECT * FROM references_data WHERE 1=1";
$params = [];

if ($search_term) {
    $query .= " AND (referencia LIKE ? OR nombre LIKE ? OR grupo_articulo LIKE ?)";
    $search_param = "%$search_term%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
}

if ($filter_grupo) {
    $query .= " AND grupo_articulo = ?";
    $params[] = $filter_grupo;
}

$query .= " ORDER BY referencia ASC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$all_products = $stmt->fetchAll();

// Ahora filtrar por categoría si es necesario (porque la categoría es dinámica)
if ($filter_cat1 && $filter_cat1 !== '') {
    $filtered_products = [];
    foreach ($all_products as $product) {
        $categorias = obtener_categorias_producto($pdo, $product['grupo_articulo']);
        if ($categorias['cat1'] === $filter_cat1) {
            $filtered_products[] = $product;
        }
    }
    $all_products = $filtered_products;
}

// Calcular paginación
$total_products = count($all_products);
$total_pages = ceil($total_products / $items_per_page);
$offset = ($current_page_num - 1) * $items_per_page;

// Obtener productos de la página actual
$products = array_slice($all_products, $offset, $items_per_page);

// Agregar categorías dinámicas a cada producto
foreach ($products as &$product) {
    $categorias = obtener_categorias_producto($pdo, $product['grupo_articulo']);
    $product['categoria1'] = $categorias['cat1'];
    $product['categoria2'] = $categorias['cat2'];
}

// Obtener lista de grupos únicos para filtro
$grupos_stmt = $pdo->query("SELECT DISTINCT grupo_articulo FROM references_data WHERE grupo_articulo IS NOT NULL AND grupo_articulo != '' ORDER BY grupo_articulo");
$grupos_disponibles = $grupos_stmt->fetchAll(PDO::FETCH_COLUMN);

// Obtener categorías únicas (dinámicas)
$categorias_unicas = [];
foreach ($grupos_disponibles as $grupo) {
    $cats = obtener_categorias_producto($pdo, $grupo);
    if (!in_array($cats['cat1'], $categorias_unicas)) {
        $categorias_unicas[] = $cats['cat1'];
    }
}
sort($categorias_unicas);

// Estadísticas
$stats_total = $pdo->query("SELECT COUNT(*) FROM references_data")->fetchColumn();
$stats_con_precio = $pdo->query("SELECT COUNT(*) FROM references_data WHERE precio > 0")->fetchColumn();
$stats_sin_precio = $pdo->query("SELECT COUNT(*) FROM references_data WHERE precio = 0 OR precio IS NULL")->fetchColumn();
$stats_con_imagen = $pdo->query("SELECT COUNT(*) FROM references_data WHERE url_imagen IS NOT NULL AND url_imagen != ''")->fetchColumn();
$stats_sin_grupo = $pdo->query("SELECT COUNT(*) FROM references_data WHERE grupo_articulo IS NULL OR grupo_articulo = ''")->fetchColumn();

?>

<div class="max-w-full mx-auto">
    <!-- Header simplificado y minimalista -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
        <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
            <div>
                <h1 class="text-2xl font-semibold text-gray-900">Gestión de Productos</h1>
                <p class="text-sm text-gray-600 mt-1">Administra el catálogo de productos</p>
            </div>
            <?php if (in_array($current_role, ['admin', 'editor'])): ?>
            <div class="flex flex-wrap items-center gap-2">
                <a href="index.php?page=gestionar_categorias" 
                   class="inline-flex items-center px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">
                    <i class="fas fa-tags mr-2"></i>Gestionar Categorías
                </a>
                <button onclick="openModal('import-modal')" 
                        class="inline-flex items-center px-3 py-2 border border-transparent rounded-md text-sm font-medium text-white bg-blue-600 hover:bg-blue-700">
                    <i class="fas fa-file-import mr-2"></i>Importar
                </button>
                <button onclick="openModal('create-modal')" 
                        class="inline-flex items-center px-3 py-2 border border-transparent rounded-md text-sm font-medium text-white bg-green-600 hover:bg-green-700">
                    <i class="fas fa-plus mr-2"></i>Nuevo Producto
                </button>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Mensajes de error/éxito -->
    <?php if ($error_message): ?>
    <div class="bg-red-50 border border-red-200 text-red-800 px-4 py-3 rounded-lg mb-4">
        <i class="fas fa-exclamation-circle mr-2"></i><?php echo htmlspecialchars($error_message); ?>
    </div>
    <?php endif; ?>

    <?php if ($success_message): ?>
    <div class="bg-green-50 border border-green-200 text-green-800 px-4 py-3 rounded-lg mb-4">
        <i class="fas fa-check-circle mr-2"></i><?php echo htmlspecialchars($success_message); ?>
    </div>
    <?php endif; ?>

    <!-- Estadísticas -->
    <div class="grid grid-cols-2 md:grid-cols-5 gap-3 mb-6">
        <div class="bg-white p-3 rounded-lg border border-gray-200">
            <div class="text-2xl font-bold text-gray-900"><?php echo number_format($stats_total); ?></div>
            <div class="text-xs text-gray-600">Total productos</div>
        </div>
        <div class="bg-white p-3 rounded-lg border border-gray-200">
            <div class="text-2xl font-bold text-green-600"><?php echo number_format($stats_con_precio); ?></div>
            <div class="text-xs text-gray-600">Con precio</div>
        </div>
        <div class="bg-white p-3 rounded-lg border border-gray-200">
            <div class="text-2xl font-bold text-red-600"><?php echo number_format($stats_sin_precio); ?></div>
            <div class="text-xs text-gray-600">Sin precio</div>
        </div>
        <div class="bg-white p-3 rounded-lg border border-gray-200">
            <div class="text-2xl font-bold text-blue-600"><?php echo number_format($stats_con_imagen); ?></div>
            <div class="text-xs text-gray-600">Con imagen</div>
        </div>
        <div class="bg-white p-3 rounded-lg border border-gray-200">
            <div class="text-2xl font-bold text-orange-600"><?php echo number_format($stats_sin_grupo); ?></div>
            <div class="text-xs text-gray-600">Sin grupo</div>
        </div>
    </div>

    <!-- Filtros -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
        <form method="GET" action="index.php" class="flex flex-wrap gap-3">
            <input type="hidden" name="page" value="importar_productos">
            
            <input type="text" name="search" value="<?php echo htmlspecialchars($search_term); ?>" 
                   placeholder="Buscar por referencia o nombre..." 
                   class="flex-1 min-w-[200px] px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
            
            <select name="grupo" class="px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                <option value="">Todos los grupos</option>
                <?php foreach ($grupos_disponibles as $grupo): ?>
                <option value="<?php echo htmlspecialchars($grupo); ?>" <?php echo $filter_grupo === $grupo ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($grupo); ?>
                </option>
                <?php endforeach; ?>
            </select>
            
            <select name="cat1" class="px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                <option value="">Todas las categorías</option>
                <?php foreach ($categorias_unicas as $cat): ?>
                <option value="<?php echo htmlspecialchars($cat); ?>" <?php echo $filter_cat1 === $cat ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($cat); ?>
                </option>
                <?php endforeach; ?>
            </select>
            
            <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm">
                <i class="fas fa-search mr-2"></i>Buscar
            </button>
            
            <?php if ($search_term || $filter_grupo || $filter_cat1): ?>
            <a href="index.php?page=importar_productos" class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400 text-sm">
                <i class="fas fa-times mr-2"></i>Limpiar
            </a>
            <?php endif; ?>
        </form>
    </div>

    <!-- Tabla de productos -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Imagen</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Referencia</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nombre</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Estado</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Grupo</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tipo</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Cat. Nivel 1</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Cat. Nivel 2</th>
                        <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Precio</th>
                        <?php if (in_array($current_role, ['admin', 'editor'])): ?>
                        <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Acciones</th>
                        <?php endif; ?>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($products as $product): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-4 py-3 whitespace-nowrap">
                            <img src="<?php echo htmlspecialchars($product['url_imagen']); ?>" 
                                 alt="<?php echo htmlspecialchars($product['referencia']); ?>"
                                 class="h-12 w-12 object-cover rounded cursor-pointer"
                                 onclick="openImageModal('<?php echo htmlspecialchars($product['url_imagen']); ?>', '<?php echo htmlspecialchars($product['nombre']); ?>')"
                                 onerror="this.onerror=null; this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDgiIGhlaWdodD0iNDgiIHZpZXdCb3g9IjAgMCA0OCA0OCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHJlY3Qgd2lkdGg9IjQ4IiBoZWlnaHQ9IjQ4IiBmaWxsPSIjRTVFN0VCIi8+CjxwYXRoIGQ9Ik0yOCAyMEwyMCAyOE0yMCAyMEwyOCAyOCIgc3Ryb2tlPSIjOUNBM0FGIiBzdHJva2Utd2lkdGg9IjIiIHN0cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVqb2luPSJyb3VuZCIvPgo8L3N2Zz4='; this.style.cursor='default'; this.onclick=null;"
                                 loading="lazy">
                        </td>
                        <td class="px-4 py-3 whitespace-nowrap text-sm font-medium text-gray-900">
                            <?php echo htmlspecialchars($product['referencia']); ?>
                        </td>
                        <td class="px-4 py-3 text-sm text-gray-900">
                            <?php echo htmlspecialchars($product['nombre']); ?>
                        </td>
                        <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                            <?php if ($product['estado']): ?>
                                <span class="px-2 py-1 text-xs font-medium rounded-full bg-blue-100 text-blue-800">
                                    <?php echo htmlspecialchars($product['estado']); ?>
                                </span>
                            <?php else: ?>
                                <span class="text-gray-400">-</span>
                            <?php endif; ?>
                        </td>
                        <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                            <?php echo htmlspecialchars($product['grupo_articulo'] ?: '-'); ?>
                        </td>
                        <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                            <?php echo htmlspecialchars($product['tipo_articulo'] ?: '-'); ?>
                        </td>
                        <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                            <span class="px-2 py-1 text-xs font-medium rounded-full <?php echo $product['categoria1'] === 'Sin definir' ? 'bg-gray-100 text-gray-600' : 'bg-green-100 text-green-800'; ?>">
                                <?php echo htmlspecialchars($product['categoria1']); ?>
                            </span>
                        </td>
                        <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                            <span class="px-2 py-1 text-xs font-medium rounded-full <?php echo $product['categoria2'] === 'Sin definir' ? 'bg-gray-100 text-gray-600' : 'bg-purple-100 text-purple-800'; ?>">
                                <?php echo htmlspecialchars($product['categoria2']); ?>
                            </span>
                        </td>
                        <td class="px-4 py-3 whitespace-nowrap text-sm text-right font-medium <?php echo $product['precio'] > 0 ? 'text-gray-900' : 'text-gray-400'; ?>">
                            $<?php echo number_format($product['precio'], 2); ?>
                        </td>
                        <?php if (in_array($current_role, ['admin', 'editor'])): ?>
                        <td class="px-4 py-3 whitespace-nowrap text-center text-sm">
                            <button onclick='editProduct(<?php echo json_encode($product); ?>)' 
                                    class="text-blue-600 hover:text-blue-900 mr-2">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button onclick="deleteProduct('<?php echo htmlspecialchars($product['referencia']); ?>')" 
                                    class="text-red-600 hover:text-red-900">
                                <i class="fas fa-trash"></i>
                            </button>
                        </td>
                        <?php endif; ?>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Paginación -->
        <?php if ($total_pages > 1): ?>
        <div class="bg-gray-50 px-4 py-3 flex items-center justify-between border-t border-gray-200">
            <div class="flex-1 flex justify-between sm:hidden">
                <?php if ($current_page_num > 1): ?>
                <a href="?page=importar_productos&page_num=<?php echo $current_page_num - 1; ?>&search=<?php echo urlencode($search_term); ?>&grupo=<?php echo urlencode($filter_grupo); ?>&cat1=<?php echo urlencode($filter_cat1); ?>" 
                   class="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                    Anterior
                </a>
                <?php endif; ?>
                <?php if ($current_page_num < $total_pages): ?>
                <a href="?page=importar_productos&page_num=<?php echo $current_page_num + 1; ?>&search=<?php echo urlencode($search_term); ?>&grupo=<?php echo urlencode($filter_grupo); ?>&cat1=<?php echo urlencode($filter_cat1); ?>" 
                   class="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                    Siguiente
                </a>
                <?php endif; ?>
            </div>
            <div class="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                <div>
                    <p class="text-sm text-gray-700">
                        Mostrando <span class="font-medium"><?php echo $offset + 1; ?></span> a 
                        <span class="font-medium"><?php echo min($offset + $items_per_page, $total_products); ?></span> de 
                        <span class="font-medium"><?php echo $total_products; ?></span> resultados
                    </p>
                </div>
                <div>
                    <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                        <?php 
                        // Mostrar primera página siempre
                        if ($current_page_num > 3): ?>
                        <a href="?page=importar_productos&page_num=1&search=<?php echo urlencode($search_term); ?>&grupo=<?php echo urlencode($filter_grupo); ?>&cat1=<?php echo urlencode($filter_cat1); ?>" 
                           class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700 hover:bg-gray-50">
                            1
                        </a>
                        <?php if ($current_page_num > 4): ?>
                        <span class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700">...</span>
                        <?php endif; ?>
                        <?php endif; ?>
                        
                        <?php 
                        // Calcular rango de páginas a mostrar
                        $start_page = max(1, $current_page_num - 2);
                        $end_page = min($total_pages, $current_page_num + 2);
                        
                        for ($i = $start_page; $i <= $end_page; $i++): ?>
                        <a href="?page=importar_productos&page_num=<?php echo $i; ?>&search=<?php echo urlencode($search_term); ?>&grupo=<?php echo urlencode($filter_grupo); ?>&cat1=<?php echo urlencode($filter_cat1); ?>" 
                           class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium <?php echo $i == $current_page_num ? 'bg-blue-50 text-blue-600 z-10' : 'text-gray-700 hover:bg-gray-50'; ?>">
                            <?php echo $i; ?>
                        </a>
                        <?php endfor; ?>
                        
                        <?php // Mostrar última página siempre
                        if ($current_page_num < $total_pages - 2): ?>
                        <?php if ($current_page_num < $total_pages - 3): ?>
                        <span class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700">...</span>
                        <?php endif; ?>
                        <a href="?page=importar_productos&page_num=<?php echo $total_pages; ?>&search=<?php echo urlencode($search_term); ?>&grupo=<?php echo urlencode($filter_grupo); ?>&cat1=<?php echo urlencode($filter_cat1); ?>" 
                           class="relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700 hover:bg-gray-50">
                            <?php echo $total_pages; ?>
                        </a>
                        <?php endif; ?>
                        
                        <?php // Botones de anterior y siguiente ?>
                        <?php if ($current_page_num > 1): ?>
                        <a href="?page=importar_productos&page_num=<?php echo $current_page_num - 1; ?>&search=<?php echo urlencode($search_term); ?>&grupo=<?php echo urlencode($filter_grupo); ?>&cat1=<?php echo urlencode($filter_cat1); ?>" 
                           class="relative inline-flex items-center px-3 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-700 hover:bg-gray-50">
                            <i class="fas fa-chevron-left"></i>
                        </a>
                        <?php endif; ?>
                        
                        <?php if ($current_page_num < $total_pages): ?>
                        <a href="?page=importar_productos&page_num=<?php echo $current_page_num + 1; ?>&search=<?php echo urlencode($search_term); ?>&grupo=<?php echo urlencode($filter_grupo); ?>&cat1=<?php echo urlencode($filter_cat1); ?>" 
                           class="relative inline-flex items-center px-3 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-700 hover:bg-gray-50">
                            <i class="fas fa-chevron-right"></i>
                        </a>
                        <?php endif; ?>
                    </nav>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal de Importación -->
<div id="import-modal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 max-w-4xl shadow-lg rounded-md bg-white">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg font-bold text-gray-900">Importar Productos</h3>
            <button onclick="closeModal('import-modal')" class="text-gray-400 hover:text-gray-500">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <form method="POST" action="index.php?page=importar_productos" onsubmit="return validateImport()">
            <input type="hidden" name="action" value="import_paste">
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-2">
                    Pega los datos del archivo Excel/CSV aquí:
                </label>
                <textarea 
                    id="paste_data"
                    name="paste_data" 
                    rows="15" 
                    class="w-full px-3 py-2 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-blue-500 focus:border-blue-500 font-mono"
                    placeholder="Copia los datos desde Excel incluyendo los encabezados y pégalos aquí..."></textarea>
            </div>
            
            <div class="bg-blue-50 border border-blue-200 rounded-md p-3 mb-4">
                <p class="text-sm text-blue-800">
                    <i class="fas fa-info-circle mr-2"></i>
                    <strong>Formato esperado:</strong> El archivo debe incluir las columnas en este orden:
                </p>
                <ol class="text-xs text-blue-700 mt-2 list-decimal list-inside">
                    <li>Código de artículo (columna 1)</li>
                    <li>Nombre de búsqueda (columna 2 - se ignora)</li>
                    <li>Nombre del artículo (columna 3)</li>
                    <li>Estado (columna 4)</li>
                    <li>Grupo de artículos (columna 5)</li>
                    <li>Tipo de artículo (columna 6)</li>
                    <li>Precio (columna 7)</li>
                </ol>
            </div>
            
            <!-- Progress indicator -->
            <div id="import-progress" class="hidden mb-4">
                <div class="bg-gray-200 rounded-full h-2">
                    <div id="progress-bar" class="bg-blue-600 h-2 rounded-full" style="width: 0%"></div>
                </div>
                <p id="progress-text" class="text-sm text-gray-600 mt-2">Procesando...</p>
            </div>
            
            <!-- Log messages area -->
            <?php if (!empty($log_messages)): ?>
            <div class="mb-4 max-h-40 overflow-y-auto bg-gray-100 rounded p-3">
                <p class="text-xs font-semibold text-gray-700 mb-2">Log de importación:</p>
                <?php foreach ($log_messages as $log): ?>
                    <p class="text-xs text-gray-600 font-mono"><?php echo htmlspecialchars($log); ?></p>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
            
            <div class="flex justify-end gap-3">
                <button type="button" onclick="closeModal('import-modal')" 
                        class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400 text-sm">
                    Cancelar
                </button>
                <button type="submit" id="import-btn"
                        class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm">
                    <i class="fas fa-upload mr-2"></i>Importar
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Modal de Crear/Editar -->
<div id="create-modal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="flex justify-between items-center mb-4">
            <h3 id="modal-title" class="text-lg font-bold text-gray-900">Nuevo Producto</h3>
            <button onclick="closeModal('create-modal')" class="text-gray-400 hover:text-gray-500">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <form method="POST" action="index.php?page=importar_productos">
            <input type="hidden" id="form-action" name="action" value="create">
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">Referencia</label>
                <input type="text" id="form-referencia" name="referencia" required 
                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">Nombre</label>
                <input type="text" id="form-nombre" name="nombre" 
                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">Grupo de Artículos</label>
                <input type="text" id="form-grupo" name="grupo_articulo" 
                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">Tipo de Artículo</label>
                <select id="form-tipo" name="tipo_articulo" 
                        class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                    <option value="Artículo">Artículo</option>
                    <option value="L.MAT">L.MAT</option>
                    <option value="Servicio">Servicio</option>
                </select>
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">Estado</label>
                <input type="text" id="form-estado" name="estado" 
                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">Precio</label>
                <input type="number" id="form-precio" name="precio" step="0.01" min="0" 
                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
            </div>
            
            <div class="flex justify-end gap-3">
                <button type="button" onclick="closeModal('create-modal')" 
                        class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400 text-sm">
                    Cancelar
                </button>
                <button type="submit" 
                        class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm">
                    <i class="fas fa-save mr-2"></i>Guardar
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Modal de Imagen -->
<div id="image-modal" class="hidden fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50" onclick="closeImageModal()">
    <div class="max-w-4xl max-h-full p-4">
        <img id="modal-image" src="" alt="" class="max-w-full max-h-full object-contain" 
             onerror="this.onerror=null; this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAwIiBoZWlnaHQ9IjQwMCIgdmlld0JveD0iMCAwIDQwMCA0MDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI0MDAiIGhlaWdodD0iNDAwIiBmaWxsPSIjRTVFN0VCIi8+CjxwYXRoIGQ9Ik0yMjAgMTgwTDE4MCAyMjBNMTgwIDE4MEwyMjAgMjIwIiBzdHJva2U9IiM5Q0EzQUYiIHN0cm9rZS13aWR0aD0iOCIgc3Ryb2tlLWxpbmVjYXA9InJvdW5kIiBzdHJva2UtbGluZWpvaW49InJvdW5kIi8+Cjx0ZXh0IHg9IjIwMCIgeT0iMjYwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMTgiIGZpbGw9IiM2QjcyODAiIHRleHQtYW5jaG9yPSJtaWRkbGUiPkltYWdlbiBubyBkaXNwb25pYmxlPC90ZXh0Pgo8L3N2Zz4=';">
        <p id="modal-caption" class="text-white text-center mt-2"></p>
    </div>
</div>

<script>
// Cache para imágenes que ya sabemos que no existen
const failedImages = new Set();

function openModal(modalId, isEdit = false) {
    document.getElementById(modalId).classList.remove('hidden');
    if (modalId === 'create-modal' && !isEdit) {
        // Solo resetear cuando es para crear nuevo, no cuando es para editar
        document.getElementById('form-action').value = 'create';
        document.getElementById('modal-title').textContent = 'Nuevo Producto';
        document.getElementById('form-referencia').value = '';
        document.getElementById('form-referencia').readOnly = false;
        document.getElementById('form-nombre').value = '';
        document.getElementById('form-grupo').value = '';
        document.getElementById('form-tipo').value = 'Artículo';
        document.getElementById('form-estado').value = '';
        document.getElementById('form-precio').value = '';
    }
}

function closeModal(modalId) {
    document.getElementById(modalId).classList.add('hidden');
}

function openImageModal(imageUrl, caption = '') {
    // Si ya sabemos que la imagen no existe, no intentar cargarla
    if (failedImages.has(imageUrl)) {
        return;
    }
    document.getElementById('modal-image').src = imageUrl;
    document.getElementById('modal-caption').textContent = caption;
    document.getElementById('image-modal').classList.remove('hidden');
}

function closeImageModal() {
    document.getElementById('image-modal').classList.add('hidden');
}

// Prevenir múltiples recargas de imágenes
document.addEventListener('DOMContentLoaded', function() {
    const images = document.querySelectorAll('img[loading="lazy"]');
    images.forEach(img => {
        // Si la imagen ya falló antes, usar placeholder directamente
        if (failedImages.has(img.src)) {
            img.src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDgiIGhlaWdodD0iNDgiIHZpZXdCb3g9IjAgMCA0OCA0OCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHJlY3Qgd2lkdGg9IjQ4IiBoZWlnaHQ9IjQ4IiBmaWxsPSIjRTVFN0VCIi8+CjxwYXRoIGQ9Ik0yOCAyMEwyMCAyOE0yMCAyMEwyOCAyOCIgc3Ryb2tlPSIjOUNBM0FGIiBzdHJva2Utd2lkdGg9IjIiIHN0cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVqb2luPSJyb3VuZCIvPgo8L3N2Zz4=';
            img.style.cursor = 'default';
            img.onclick = null;
        }
        
        // Marcar imágenes que fallan para no reintentarlas
        img.addEventListener('error', function() {
            failedImages.add(this.src);
        });
    });
});

function editProduct(productJson) {
    try {
        const product = typeof productJson === 'string' ? JSON.parse(productJson) : productJson;
        
        // Configurar el formulario para edición
        document.getElementById('form-action').value = 'edit';
        document.getElementById('modal-title').textContent = 'Editar Producto';
        document.getElementById('form-referencia').value = product.referencia || '';
        document.getElementById('form-referencia').readOnly = true;
        document.getElementById('form-nombre').value = product.nombre || '';
        document.getElementById('form-grupo').value = product.grupo_articulo || '';
        document.getElementById('form-tipo').value = product.tipo_articulo || 'Artículo';
        document.getElementById('form-estado').value = product.estado || '';
        document.getElementById('form-precio').value = product.precio || 0;
        
        // Abrir el modal indicando que es para editar
        openModal('create-modal', true);
    } catch (error) {
        console.error('Error al editar producto:', error);
        alert('Error al cargar los datos del producto');
    }
}

function deleteProduct(referencia) {
    if (confirm(`¿Estás seguro de eliminar el producto ${referencia}?`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = 'index.php?page=importar_productos';
        
        const actionInput = document.createElement('input');
        actionInput.type = 'hidden';
        actionInput.name = 'action';
        actionInput.value = 'delete';
        
        const refInput = document.createElement('input');
        refInput.type = 'hidden';
        refInput.name = 'referencia';
        refInput.value = referencia;
        
        form.appendChild(actionInput);
        form.appendChild(refInput);
        document.body.appendChild(form);
        form.submit();
    }
}

function validateImport() {
    const pasteData = document.getElementById('paste_data').value.trim();
    if (!pasteData) {
        alert('Por favor, pega los datos antes de importar.');
        return false;
    }
    
    // Mostrar indicador de progreso
    const progressDiv = document.getElementById('import-progress');
    const importBtn = document.getElementById('import-btn');
    
    if (confirm('¿Deseas importar estos datos? Los productos existentes serán actualizados.')) {
        progressDiv.classList.remove('hidden');
        importBtn.disabled = true;
        importBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Importando...';
        return true;
    }
    
    return false;
}
</script>