<?php
/**
 * Script de actualización automática del catálogo público
 * Se ejecuta cada 6 horas para mantener los datos actualizados
 */

// Configuración de base de datos
require_once '/home/crusardi/config_crusardi/database.php';

// Configurar charset UTF-8
if ($pdo) {
    $pdo->exec("SET NAMES 'utf8mb4'");
    $pdo->exec("SET CHARACTER SET utf8mb4");
    $pdo->exec("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");
}

date_default_timezone_set('America/Bogota');

// Log de ejecución
$log_file = __DIR__ . '/logs/auto_update.log';
$timestamp = date('Y-m-d H:i:s');

function writeLog($message) {
    global $log_file, $timestamp;
    $log_entry = "[$timestamp] $message" . PHP_EOL;
    file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
}

try {
    writeLog("Iniciando actualización automática del catálogo público");
    
    // 1. Ejecutar consolidado de stock
    writeLog("Ejecutando consolidado de stock...");
    $stmt = $pdo->prepare("CALL consolidar_stock_crusardi()");
    $stmt->execute();
    $stmt->closeCursor();
    writeLog("Consolidado de stock completado");
    
    // 2. Verificar que la tabla cache esté actualizada
    $check_query = "SELECT COUNT(*) as total FROM stock_calculado_cache WHERE stock_total > 0 AND precio > 0";
    $result = $pdo->query($check_query)->fetch(PDO::FETCH_ASSOC);
    $productos_disponibles = $result['total'];
    
    writeLog("Productos disponibles en cache: $productos_disponibles");
    
    if ($productos_disponibles == 0) {
        throw new Exception("No hay productos disponibles en el cache");
    }
    
    // 3. Actualizar timestamp de última actualización
    $update_time = date('Y-m-d H:i:s');
    $timestamp_file = __DIR__ . '/logs/ultima_actualizacion.txt';
    file_put_contents($timestamp_file, $update_time);
    
    // 4. Generar estadísticas de actualización
    $stats_query = "
        SELECT 
            COUNT(*) as total_productos,
            SUM(stock_total) as total_unidades,
            SUM(CASE WHEN tipo = 'kit' THEN 1 ELSE 0 END) as total_kits,
            SUM(CASE WHEN tipo = 'articulo' THEN 1 ELSE 0 END) as total_articulos,
            SUM(CASE WHEN es_imperfecto = 1 THEN 1 ELSE 0 END) as total_imperfectos,
            AVG(precio) as precio_promedio
        FROM stock_calculado_cache 
        WHERE stock_total > 0 AND precio > 0
    ";
    
    $stats = $pdo->query($stats_query)->fetch(PDO::FETCH_ASSOC);
    
    // 5. Guardar estadísticas en archivo JSON para consulta rápida
    $stats_file = __DIR__ . '/logs/estadisticas.json';
    file_put_contents($stats_file, json_encode($stats, JSON_PRETTY_PRINT));
    
    // 6. Verificar que el catálogo público sea accesible
    $catalogo_url = 'https://stock.crusardi.net/catalogo_publico/';
    $context = stream_context_create([
        'http' => [
            'timeout' => 30,
            'method' => 'GET'
        ]
    ]);
    
    $response = @file_get_contents($catalogo_url, false, $context);
    
    if ($response === false) {
        writeLog("ADVERTENCIA: No se pudo verificar la accesibilidad del catálogo público");
    } else {
        writeLog("Catálogo público verificado correctamente");
    }
    
    // 7. Limpiar logs antiguos (mantener solo últimos 30 días)
    $log_files = glob(__DIR__ . '/logs/*.log');
    $cutoff_date = strtotime('-30 days');
    
    foreach ($log_files as $file) {
        if (filemtime($file) < $cutoff_date) {
            unlink($file);
            writeLog("Log antiguo eliminado: " . basename($file));
        }
    }
    
    // 8. Resumen de la actualización
    $summary = [
        'timestamp' => $update_time,
        'productos_disponibles' => $productos_disponibles,
        'total_unidades' => $stats['total_unidades'],
        'total_kits' => $stats['total_kits'],
        'total_articulos' => $stats['total_articulos'],
        'total_imperfectos' => $stats['total_imperfectos'],
        'precio_promedio' => round($stats['precio_promedio'], 0),
        'status' => 'success'
    ];
    
    writeLog("Actualización completada exitosamente");
    writeLog("Resumen: " . json_encode($summary));
    
    // 9. Enviar notificación por email (opcional)
    if (function_exists('mail')) {
        $to = 'tecnologia@crusardi.net';
        $subject = 'Catálogo Público Actualizado - ' . $update_time;
        $message = "
        El catálogo público de BoConcept Crusardi ha sido actualizado exitosamente.
        
        Resumen de la actualización:
        - Productos disponibles: " . number_format($productos_disponibles) . "
        - Total unidades: " . number_format($stats['total_unidades']) . "
        - Kits: " . number_format($stats['total_kits']) . "
        - Artículos: " . number_format($stats['total_articulos']) . "
        - Oportunidades: " . number_format($stats['total_imperfectos']) . "
        - Precio promedio: $" . number_format($stats['precio_promedio'], 0, ',', '.') . "
        
        URL del catálogo: https://stock.crusardi.net/catalogo_publico/
        
        Actualizado el: $update_time
        ";
        
        $headers = [
            'From: sistema@crusardi.net',
            'Reply-To: tecnologia@crusardi.net',
            'Content-Type: text/plain; charset=UTF-8'
        ];
        
        @mail($to, $subject, $message, implode("\r\n", $headers));
        writeLog("Notificación por email enviada");
    }
    
    echo json_encode([
        'success' => true,
        'timestamp' => $update_time,
        'summary' => $summary
    ]);
    
} catch (Exception $e) {
    writeLog("ERROR: " . $e->getMessage());
    
    // Enviar notificación de error
    if (function_exists('mail')) {
        $to = 'tecnologia@crusardi.net';
        $subject = 'ERROR - Actualización Catálogo Público';
        $message = "
        Error en la actualización automática del catálogo público:
        
        Error: " . $e->getMessage() . "
        Timestamp: $timestamp
        
        Por favor revisar los logs para más detalles.
        ";
        
        $headers = [
            'From: sistema@crusardi.net',
            'Reply-To: tecnologia@crusardi.net',
            'Content-Type: text/plain; charset=UTF-8'
        ];
        
        @mail($to, $subject, $message, implode("\r\n", $headers));
    }
    
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'timestamp' => $timestamp
    ]);
    
    http_response_code(500);
}
?>
