<?php
// pages/importar_stock.php

$error_message = null;
$success_message = null;
$log_messages = [];

// Verificar si la función ya existe antes de declararla
if (!function_exists('normalize_header')) {
    function normalize_header($str) {
        $str = strtolower(trim($str));
        $unwanted_array = ['á' => 'a', 'é' => 'e', 'í' => 'i', 'ó' => 'o', 'ú' => 'u', 'ñ' => 'n', ' ' => '_', "\t" => '_', "\n" => '_', "\r" => '_'];
        $str = strtr($str, $unwanted_array);
        return preg_replace('/_+/', '_', $str);
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && in_array($_SESSION['user_role'], ['admin', 'editor'])) {
    $startTime = microtime(true);
    $log_messages[] = "[INFO] " . date('H:i:s') . " - Proceso de importación iniciado.";

    $sources = ['bw1' => 'BW1', 'bb1' => 'BB1', 'bb2' => 'BB2', 'md1' => 'MD1'];
    $dataFound = false;

    foreach ($sources as $key => $bodega) {
        if ((isset($_FILES[$key.'_file']) && $_FILES[$key.'_file']['error'] === UPLOAD_ERR_OK) || !empty(trim($_POST[$key.'_text'] ?? ''))) {
            $dataFound = true;
            break;
        }
    }

    if ($dataFound) {
        try {
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Conectando a la base de datos...";
            
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Borrando stock anterior (TRUNCATE TABLE)...";
            $pdo->exec("TRUNCATE TABLE stock_data");
            $log_messages[] = "[OK] " . date('H:i:s') . " - Tabla 'stock_data' vaciada correctamente.";

            $pdo->beginTransaction();
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Transacción de inserción iniciada.";

            // Actualizar con fecha de Colombia
            $stmt = $pdo->prepare("INSERT INTO stock_data (referencia, bodega, almacen, cantidad, fecha_actualizacion) VALUES (?, ?, ?, ?, NOW())");
            $totalRowsImported = 0;

            foreach ($sources as $key => $bodega) {
                $fileHandle = null;
                $source_type = "vacío";

                if (isset($_FILES[$key.'_file']) && $_FILES[$key.'_file']['error'] === UPLOAD_ERR_OK) {
                    $fileHandle = fopen($_FILES[$key.'_file']['tmp_name'], 'r');
                    $source_type = "archivo '" . htmlspecialchars($_FILES[$key.'_file']['name']) . "'";
                }
                elseif (!empty(trim($_POST[$key.'_text'] ?? ''))) {
                    $fileHandle = fopen('php://memory', 'r+');
                    fwrite($fileHandle, $_POST[$key.'_text']);
                    rewind($fileHandle);
                    $source_type = "texto pegado";
                }

                if ($fileHandle) {
                    $log_messages[] = "[INFO] " . date('H:i:s') . " - Procesando bodega '$bodega' desde $source_type...";
                    
                    $firstLine = fgets($fileHandle);
                    if (!$firstLine) {
                         $log_messages[] = "[WARN] " . date('H:i:s') . " - La fuente para '$bodega' está vacía. Omitiendo.";
                         fclose($fileHandle);
                         continue;
                    }
                    
                    // Detectar delimitador
                    $delimiter = "\t"; // Por defecto tabulador (Axapta)
                    if (strpos($firstLine, "\t") === false && strpos($firstLine, ",") !== false) {
                        $delimiter = ",";
                    }
                    
                    rewind($fileHandle);
                    $header = fgetcsv($fileHandle, 0, $delimiter);
                    if (!$header) {
                        $log_messages[] = "[WARN] " . date('H:i:s') . " - No se pudo leer la cabecera para '$bodega'. Omitiendo.";
                        fclose($fileHandle);
                        continue;
                    }
                    
                    $normalized_headers = array_map('normalize_header', $header);
                    
                    // Buscar columnas - más flexible
                    $refIndex = false;
                    $almacenIndex = false;
                    $qtyIndex = false;
                    
                    // Posibles nombres para referencia
                    $ref_names = ['codigo_de_articulo', 'referencia', 'itemid', 'codigo', 'articulo', 'item'];
                    foreach ($ref_names as $name) {
                        $idx = array_search($name, $normalized_headers);
                        if ($idx !== false) {
                            $refIndex = $idx;
                            break;
                        }
                    }
                    
                    // Posibles nombres para almacén
                    $almacen_names = ['almacen', 'bodega', 'inventlocationid', 'ubicacion', 'warehouse'];
                    foreach ($almacen_names as $name) {
                        $idx = array_search($name, $normalized_headers);
                        if ($idx !== false) {
                            $almacenIndex = $idx;
                            break;
                        }
                    }
                    
                    // Posibles nombres para cantidad
                    $qty_names = ['fisico_disponible', 'inventario_fisico', 'physicalinvent', 'cantidad', 'stock', 'qty'];
                    foreach ($qty_names as $name) {
                        $idx = array_search($name, $normalized_headers);
                        if ($idx !== false) {
                            $qtyIndex = $idx;
                            break;
                        }
                    }
                    
                    if ($refIndex === false || $almacenIndex === false || $qtyIndex === false) {
                        $log_messages[] = "[ERROR] " . date('H:i:s') . " - Columnas requeridas no encontradas en '$bodega'. Headers encontrados: " . implode(', ', $normalized_headers);
                        fclose($fileHandle);
                        continue;
                    }

                    $rowsInSource = 0;
                    $lineNumber = 1; // Empezamos en 1 porque ya leímos el header
                    
                    while (($data = fgetcsv($fileHandle, 0, $delimiter)) !== FALSE) {
                        $lineNumber++;
                        
                        if (count($data) < max($refIndex, $almacenIndex, $qtyIndex) + 1) {
                            continue;
                        }
                        
                        $ref = trim($data[$refIndex]);
                        $almacen = strtoupper(trim($data[$almacenIndex]));
                        
                        // Procesar cantidad - manejar formato con coma decimal
                        $cantidad_str = $data[$qtyIndex];
                        $cantidad_str = str_replace('.', '', $cantidad_str); // Quitar puntos de miles
                        $cantidad_str = str_replace(',', '.', $cantidad_str); // Cambiar coma por punto decimal
                        $cantidad = floatval($cantidad_str);
                        
                        if ($ref && $almacen && $cantidad > 0) {
                            $stmt->execute([$ref, $bodega, $almacen, $cantidad]);
                            $rowsInSource++;
                        }
                    }
                    
                    $log_messages[] = "[OK] " . date('H:i:s') . " - Bodega '$bodega' procesada. Se importaron $rowsInSource registros de $lineNumber líneas.";
                    $totalRowsImported += $rowsInSource;
                    fclose($fileHandle);
                }
            }
            
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Confirmando transacción (COMMIT)...";
            $pdo->commit();
            
            // Borrar caché si existe
            @unlink(__DIR__ . '/../cache/public_catalog_data.json');
            $log_messages[] = "[INFO] " . date('H:i:s') . " - Caché del catálogo público invalidada.";
            
            // Actualizar fecha de última importación (si la tabla existe)
            try {
                $stmt_config = $pdo->prepare("INSERT INTO configuracion (clave, valor) VALUES ('ultima_importacion_stock', NOW()) ON DUPLICATE KEY UPDATE valor = NOW()");
                $stmt_config->execute();
            } catch (PDOException $e) {
                // La tabla configuracion no existe, no es crítico
                $log_messages[] = "[INFO] " . date('H:i:s') . " - Tabla 'configuracion' no existe, continuando sin actualizar fecha.";
            }
            
            $log_messages[] = "[OK] " . date('H:i:s') . " - Transacción completada.";
            $endTime = microtime(true);
            $executionTime = round($endTime - $startTime, 2);
            $success_message = "Importación completada en $executionTime segundos. Se borró el inventario anterior y se cargaron $totalRowsImported registros nuevos.";
            $log_messages[] = "[SUCCESS] " . date('H:i:s') . " - $success_message";

        } catch (Exception $e) {
            if ($pdo && $pdo->inTransaction()) {
                $pdo->rollBack();
                $log_messages[] = "[WARN] " . date('H:i:s') . " - Transacción revertida (ROLLBACK) debido a un error.";
            }
            $error_message = "Error durante la importación: " . $e->getMessage();
            $log_messages[] = "[ERROR] " . date('H:i:s') . " - $error_message";
        }
    } else {
        $error_message = "No se proporcionaron datos. Por favor, sube un archivo o pega texto en al menos un campo.";
        $log_messages[] = "[WARN] " . date('H:i:s') . " - Intento de importación sin datos.";
    }
}

// Obtener estadísticas actuales
try {
    $stats = [
        'total_bodegas' => $pdo->query("SELECT COUNT(DISTINCT bodega) FROM stock_data")->fetchColumn() ?: 0,
        'total_referencias' => $pdo->query("SELECT COUNT(DISTINCT referencia) FROM stock_data")->fetchColumn() ?: 0,
        'total_unidades' => $pdo->query("SELECT COALESCE(SUM(cantidad), 0) FROM stock_data")->fetchColumn() ?: 0,
        'ultima_actualizacion' => $pdo->query("SELECT MAX(fecha_actualizacion) FROM stock_data")->fetchColumn() ?: null
    ];
    
    // Stock por bodega
    $stock_por_bodega = $pdo->query("
        SELECT bodega, COUNT(DISTINCT referencia) as refs, SUM(cantidad) as total 
        FROM stock_data 
        GROUP BY bodega 
        ORDER BY bodega
    ")->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    $stats = [
        'total_bodegas' => 0,
        'total_referencias' => 0,
        'total_unidades' => 0,
        'ultima_actualizacion' => null
    ];
    $stock_por_bodega = [];
}
?>

<div class="container mx-auto">
    <header class="flex items-center justify-between gap-4 mb-6">
        <div class="flex items-center gap-4">
            <div class="bg-blue-600 text-white p-3 rounded-lg shadow-md"><i class="fas fa-database fa-2x"></i></div>
            <div>
                <h1 class="text-3xl font-bold text-gray-800">Importador de Stock por Tienda</h1>
                <p class="text-gray-500">Cargar el inventario diario desde Axapta para cada tienda</p>
            </div>
        </div>
    </header>

    <?php if ($success_message): ?>
        <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6 rounded-r-lg" role="alert">
            <p class="font-bold">Éxito</p>
            <p><?php echo htmlspecialchars($success_message); ?></p>
        </div>
    <?php endif; ?>
    
    <?php if ($error_message): ?>
         <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6 rounded-r-lg" role="alert">
             <p class="font-bold">Error</p>
             <p><?php echo htmlspecialchars($error_message); ?></p>
         </div>
    <?php endif; ?>

    <!-- Estadísticas -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-white rounded-lg shadow p-4 border-t-4 border-blue-500">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm text-gray-600">Tiendas Activas</p>
                    <p class="text-2xl font-bold"><?php echo number_format($stats['total_bodegas']); ?></p>
                </div>
                <i class="fas fa-store text-blue-500 text-2xl"></i>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow p-4 border-t-4 border-green-500">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm text-gray-600">Referencias Totales</p>
                    <p class="text-2xl font-bold"><?php echo number_format($stats['total_referencias']); ?></p>
                </div>
                <i class="fas fa-barcode text-green-500 text-2xl"></i>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow p-4 border-t-4 border-purple-500">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm text-gray-600">Unidades Totales</p>
                    <p class="text-2xl font-bold"><?php echo number_format($stats['total_unidades']); ?></p>
                </div>
                <i class="fas fa-cubes text-purple-500 text-2xl"></i>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow p-4 border-t-4 border-orange-500">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm text-gray-600">Última Actualización</p>
                    <p class="text-lg font-semibold">
                        <?php 
                        if ($stats['ultima_actualizacion']) {
                            echo date('d/m/Y H:i', strtotime($stats['ultima_actualizacion']));
                        } else {
                            echo 'Nunca';
                        }
                        ?>
                    </p>
                </div>
                <i class="fas fa-clock text-orange-500 text-2xl"></i>
            </div>
        </div>
    </div>

    <?php if (!empty($stock_por_bodega)): ?>
    <div class="bg-white rounded-lg shadow p-4 mb-6">
        <h3 class="font-semibold text-gray-700 mb-3">Stock por Tienda</h3>
        <div class="grid grid-cols-2 md:grid-cols-4 gap-3">
            <?php foreach ($stock_por_bodega as $bodega): ?>
            <div class="bg-gray-50 p-3 rounded">
                <p class="font-bold text-gray-800"><?php echo $bodega['bodega']; ?></p>
                <p class="text-sm text-gray-600"><?php echo number_format($bodega['refs']); ?> refs</p>
                <p class="text-sm text-gray-600"><?php echo number_format($bodega['total']); ?> uds</p>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>

    <?php if (in_array($_SESSION['user_role'], ['admin', 'editor'])): ?>
    <div class="bg-white p-6 rounded-xl shadow-lg">
        <form method="POST" enctype="multipart/form-data" id="import-form" action="index.php?page=importar_stock">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                <?php foreach(['bw1' => 'BW1 - CEDI', 'bb1' => 'BB1 - CALLE 81', 'bb2' => 'BB2 - AVENIDA 19 ', 'md1' => 'MD1 - MILLA DE ORO'] as $key => $label): ?>
                <div class="bg-gray-50 p-4 rounded-lg border border-gray-200">
                    <h3 class="font-bold text-lg text-gray-800 mb-3"><?php echo $label; ?></h3>
                    <div class="flex mb-3 rounded-md border bg-gray-200 p-1">
                        <button type="button" data-source="<?php echo $key; ?>" data-tab="paste" class="tab-button flex-1 p-2 text-sm rounded-md active bg-blue-600 text-white">
                            <i class="fas fa-paste mr-1"></i>Pegar Texto
                        </button>
                        <button type="button" data-source="<?php echo $key; ?>" data-tab="file" class="tab-button flex-1 p-2 text-sm rounded-md">
                            <i class="fas fa-file mr-1"></i>Archivo
                        </button>
                    </div>
                    
                    <div id="paste-container-<?php echo $key; ?>">
                         <textarea name="<?php echo $key; ?>_text" 
                                   class="w-full h-32 p-2 border rounded-md font-mono text-xs" 
                                   placeholder="Pega aquí los datos copiados desde Axapta..."></textarea>
                         <p class="text-xs text-gray-500 mt-1">
                             <i class="fas fa-info-circle"></i> Copia desde Excel/Axapta incluyendo los encabezados
                         </p>
                    </div>
                    
                    <div id="file-container-<?php echo $key; ?>" class="hidden">
                        <label for="<?php echo $key; ?>_file" class="drop-zone bg-white rounded-md p-4 cursor-pointer block">
                            <input type="file" name="<?php echo $key; ?>_file" id="<?php echo $key; ?>_file" class="hidden" accept=".csv,.txt">
                            <div class="flex flex-col items-center justify-center">
                                <i class="fas fa-file-csv fa-3x text-gray-400"></i>
                                <span class="text-sm mt-2">Arrastra archivo o haz clic</span>
                                <span class="file-name text-xs text-gray-500 mt-1 italic" id="file-name-<?php echo $key; ?>"></span>
                            </div>
                        </label>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>

            <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-triangle text-yellow-400"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm text-yellow-700">
                            <strong>Importante:</strong> Esta acción borrará TODO el stock anterior y lo reemplazará con los datos nuevos.
                        </p>
                    </div>
                </div>
            </div>

            <div class="text-center">
                 <button type="submit" id="submit-btn" class="bg-blue-600 text-white font-bold py-3 px-8 rounded-lg shadow-lg hover:bg-blue-700 disabled:bg-gray-400 w-64">
                    <span class="btn-text"><i class="fas fa-upload mr-2"></i>Borrar y Reemplazar Stock</span>
                    <i class="fas fa-spinner fa-spin hidden spinner"></i>
                </button>
            </div>
        </form>
    </div>
    <?php endif; ?>

    <?php if (!empty($log_messages)): ?>
    <div class="mt-8">
        <h3 class="text-lg font-semibold text-gray-700 mb-2">Consola de Proceso</h3>
        <div class="bg-gray-900 text-white font-mono text-sm p-4 rounded-lg h-64 overflow-y-auto">
            <?php foreach ($log_messages as $msg): ?>
                <p class="<?php
                    if (strpos($msg, '[OK]') !== false) echo 'text-green-400';
                    elseif (strpos($msg, '[SUCCESS]') !== false) echo 'text-green-300 font-bold';
                    elseif (strpos($msg, '[WARN]') !== false) echo 'text-yellow-400';
                    elseif (strpos($msg, '[ERROR]') !== false) echo 'text-red-400';
                    else echo 'text-gray-300';
                ?>"><?php echo htmlspecialchars($msg); ?></p>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
    // Manejo de pestañas
    document.querySelectorAll('.tab-button').forEach(button => {
        button.addEventListener('click', () => {
            const source = button.dataset.source;
            const tab = button.dataset.tab;
            
            // Resetear botones de esta fuente
            document.querySelectorAll(`.tab-button[data-source="${source}"]`).forEach(btn => {
                btn.classList.remove('active', 'bg-blue-600', 'text-white');
            });
            
            // Activar botón actual
            button.classList.add('active', 'bg-blue-600', 'text-white');
            
            // Mostrar/ocultar contenedores
            document.getElementById(`file-container-${source}`).classList.toggle('hidden', tab !== 'file');
            document.getElementById(`paste-container-${source}`).classList.toggle('hidden', tab !== 'paste');
        });
    });

    // Manejo de archivos
    document.querySelectorAll('input[type="file"]').forEach(input => {
        const source = input.name.replace('_file', '');
        const fileNameSpan = document.getElementById(`file-name-${source}`);
        const dropZone = input.closest('.drop-zone');
        
        if (!dropZone) return;

        input.addEventListener('change', () => {
            fileNameSpan.textContent = input.files.length > 0 ? input.files[0].name : '';
        });
        
        // Drag and drop
        ['dragenter', 'dragover'].forEach(eventName => {
            dropZone.addEventListener(eventName, (e) => {
                e.preventDefault();
                dropZone.classList.add('bg-blue-50', 'border-blue-500');
            });
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, (e) => {
                e.preventDefault();
                dropZone.classList.remove('bg-blue-50', 'border-blue-500');
            });
        });
        
        dropZone.addEventListener('drop', (e) => {
            if (e.dataTransfer.files.length) {
                input.files = e.dataTransfer.files;
                fileNameSpan.textContent = e.dataTransfer.files[0].name;
            }
        });
    });

    // Spinner al enviar
    const importForm = document.getElementById('import-form');
    if(importForm) {
        importForm.addEventListener('submit', () => {
            const submitBtn = document.getElementById('submit-btn');
            submitBtn.disabled = true;
            submitBtn.querySelector('.btn-text').classList.add('hidden');
            submitBtn.querySelector('.spinner').classList.remove('hidden');
        });
    }
});
</script>

<style>
.tab-button { transition: all .2s ease; cursor: pointer; }
.tab-button:hover { background-color: #e5e7eb; }
.tab-button.active { background-color: #2563eb !important; color: #fff !important; }
.drop-zone { border: 2px dashed #cbd5e1; transition: .2s; }
.drop-zone:hover { background-color: #f9fafb; }
.spinner { animation: spin 1s linear infinite; }
@keyframes spin { to { transform: rotate(360deg); } }
</style>