<?php

// Configuración de zona horaria
date_default_timezone_set('America/Bogota');

// Incluir configuración de base de datos
require_once '/home/crusardi/config_crusardi/database.php';

// Configuración de correo desde archivo de configuración
$config_correo = [
    'smtp_host' => $config['smtp_host'] ?? 'smtp.gmail.com',
    'smtp_port' => $config['smtp_port'] ?? 587,
    'smtp_username' => $config['smtp_username'] ?? 'alertas@crusardi.net',
    'smtp_password' => $config['smtp_password'] ?? '',
    'from_email' => $config['from_email'] ?? 'alertas@crusardi.net',
    'from_name' => $config['from_name'] ?? 'Sistema de Alertas - Crusardi',
    'to_emails' => $config['to_emails'] ?? [
        'comercial@crusardi.net',
        'admin@crusardi.net'
    ]
];

// Verificar que las credenciales estén configuradas
if (empty($config_correo['smtp_username']) || empty($config_correo['smtp_password'])) {
    die("❌ Error: Las credenciales de correo no están configuradas en database.php\n");
}

// Función para enviar correo
function enviarCorreo($asunto, $mensaje, $config) {
    try {
        // Usar PHPMailer si está disponible
        if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
            return enviarConPHPMailer($asunto, $mensaje, $config);
        } else {
            // Fallback a mail() nativo
            return enviarConMailNativo($asunto, $mensaje, $config);
        }
    } catch (Exception $e) {
        error_log("Error enviando correo: " . $e->getMessage());
        return false;
    }
}

// Función para enviar con PHPMailer
function enviarConPHPMailer($asunto, $mensaje, $config) {
    $mail = new PHPMailer\PHPMailer\PHPMailer(true);
    
    try {
        $mail->isSMTP();
        $mail->Host = $config['smtp_host'];
        $mail->SMTPAuth = true;
        $mail->Username = $config['smtp_username'];
        $mail->Password = $config['smtp_password'];
        $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port = $config['smtp_port'];
        $mail->CharSet = 'UTF-8';
        
        $mail->setFrom($config['from_email'], $config['from_name']);
        
        foreach ($config['to_emails'] as $email) {
            $mail->addAddress($email);
        }
        
        $mail->isHTML(true);
        $mail->Subject = $asunto;
        $mail->Body = $mensaje;
        
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log("PHPMailer error: " . $e->getMessage());
        return false;
    }
}

// Función para enviar con mail() nativo
function enviarConMailNativo($asunto, $mensaje, $config) {
    $headers = [
        'MIME-Version: 1.0',
        'Content-type: text/html; charset=UTF-8',
        'From: ' . $config['from_name'] . ' <' . $config['from_email'] . '>',
        'Reply-To: ' . $config['from_email'],
        'X-Mailer: PHP/' . phpversion()
    ];
    
    $mensaje_completo = "
    <html>
    <head>
        <title>$asunto</title>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: #dc2626; color: white; padding: 20px; text-align: center; }
            .content { background: #f9fafb; padding: 20px; }
            .alert { background: #fef2f2; border: 1px solid #fecaca; padding: 15px; margin: 15px 0; border-radius: 5px; }
            .warning { background: #fffbeb; border: 1px solid #fed7aa; padding: 15px; margin: 15px 0; border-radius: 5px; }
            .footer { background: #f3f4f6; padding: 15px; text-align: center; font-size: 12px; color: #6b7280; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h1>🚨 ALERTA DE STOCK</h1>
                <p>Sistema de Monitoreo - Crusardi</p>
            </div>
            <div class='content'>
                $mensaje
            </div>
            <div class='footer'>
                <p>Este correo fue generado automáticamente por el sistema de alertas de Crusardi</p>
                <p>Fecha: " . date('d/m/Y H:i:s') . "</p>
            </div>
        </div>
    </body>
    </html>";
    
    foreach ($config['to_emails'] as $email) {
        mail($email, $asunto, $mensaje_completo, implode("\r\n", $headers));
    }
    
    return true;
}

// Función para verificar stock por tienda
function verificarStockPorTienda($pdo) {
    $tiendas = ['BB1', 'BB2', 'MD1', 'BW1'];
    $alertas = [];
    
    foreach ($tiendas as $tienda) {
        try {
            // Contar referencias por tienda
            $stmt = $pdo->prepare("
                SELECT 
                    COUNT(DISTINCT referencia) as total_referencias,
                    SUM(cantidad) as total_unidades,
                    MAX(fecha_actualizacion) as ultima_actualizacion
                FROM stock_data 
                WHERE bodega = ?
            ");
            $stmt->execute([$tienda]);
            $resultado = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($resultado['total_referencias'] == 0) {
                $alertas[] = [
                    'tipo' => 'CRITICA',
                    'tienda' => $tienda,
                    'mensaje' => "La tienda $tienda no tiene NINGUNA referencia de stock",
                    'detalles' => "Total referencias: 0, Total unidades: 0"
                ];
            } elseif ($resultado['total_referencias'] < 100) {
                $alertas[] = [
                    'tipo' => 'ADVERTENCIA',
                    'tienda' => $tienda,
                    'mensaje' => "La tienda $tienda tiene muy pocas referencias",
                    'detalles' => "Total referencias: {$resultado['total_referencias']}, Total unidades: {$resultado['total_unidades']}"
                ];
            }
            
            // Verificar última actualización por tienda
            if ($resultado['ultima_actualizacion']) {
                $ultima_actualizacion = new DateTime($resultado['ultima_actualizacion']);
                $ahora = new DateTime();
                $diferencia = $ahora->diff($ultima_actualizacion);
                
                if ($diferencia->days > 1) {
                    $alertas[] = [
                        'tipo' => 'ADVERTENCIA',
                        'tienda' => $tienda,
                        'mensaje' => "La tienda $tienda no se actualiza desde hace {$diferencia->days} días",
                        'detalles' => "Última actualización: " . $resultado['ultima_actualizacion']
                    ];
                }
            }
            
        } catch (Exception $e) {
            $alertas[] = [
                'tipo' => 'ERROR',
                'tienda' => $tienda,
                'mensaje' => "Error verificando la tienda $tienda",
                'detalles' => $e->getMessage()
            ];
        }
    }
    
    return $alertas;
}

// Función para verificar última actualización general
function verificarUltimaActualizacion($pdo) {
    try {
        // Verificar stock_data
        $stmt = $pdo->prepare("
            SELECT 
                MAX(fecha_actualizacion) as ultima_stock,
                COUNT(DISTINCT referencia) as total_referencias
            FROM stock_data
        ");
        $stmt->execute();
        $stock_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Verificar stock_calculado_cache
        $stmt = $pdo->prepare("
            SELECT 
                MAX(ultima_actualizacion) as ultima_calculado,
                COUNT(DISTINCT codigo) as total_productos
            FROM stock_calculado_cache
        ");
        $stmt->execute();
        $stock_calculado = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $alertas = [];
        
        // Verificar stock_data
        if ($stock_data['ultima_stock']) {
            $ultima_stock = new DateTime($stock_data['ultima_stock']);
            $ahora = new DateTime();
            $diferencia_stock = $ahora->diff($ultima_stock);
            
            if ($diferencia_stock->days > 1) {
                $alertas[] = [
                    'tipo' => 'CRITICA',
                    'seccion' => 'Stock Físico',
                    'mensaje' => "El stock físico no se actualiza desde hace {$diferencia_stock->days} días",
                    'detalles' => "Última actualización: {$stock_data['ultima_stock']}, Total referencias: {$stock_data['total_referencias']}"
                ];
            }
        } else {
            $alertas[] = [
                'tipo' => 'CRITICA',
                'seccion' => 'Stock Físico',
                'mensaje' => "No hay registros de stock físico",
                'detalles' => "La tabla stock_data está vacía o no tiene fechas"
            ];
        }
        
        // Verificar stock_calculado_cache
        if ($stock_calculado['ultima_calculado']) {
            $ultima_calculado = new DateTime($stock_calculado['ultima_calculado']);
            $ahora = new DateTime();
            $diferencia_calculado = $ahora->diff($ultima_calculado);
            
            if ($diferencia_calculado->days > 1) {
                $alertas[] = [
                    'tipo' => 'CRITICA',
                    'seccion' => 'Stock Calculado',
                    'mensaje' => "El stock calculado no se actualiza desde hace {$diferencia_calculado->days} días",
                    'detalles' => "Última actualización: {$stock_calculado['ultima_calculado']}, Total productos: {$stock_calculado['total_productos']}"
                ];
            }
        } else {
            $alertas[] = [
                'tipo' => 'CRITICA',
                'seccion' => 'Stock Calculado',
                'mensaje' => "No hay registros de stock calculado",
                'detalles' => "La tabla stock_calculado_cache está vacía o no tiene fechas"
            ];
        }
        
        return $alertas;
        
    } catch (Exception $e) {
        return [[
            'tipo' => 'ERROR',
            'seccion' => 'Verificación General',
            'mensaje' => "Error verificando actualizaciones",
            'detalles' => $e->getMessage()
        ]];
    }
}

// Función principal de verificación
function ejecutarVerificaciones($pdo, $config_correo) {
    $alertas = [];
    
    // Verificar stock por tienda
    $alertas_tiendas = verificarStockPorTienda($pdo);
    $alertas = array_merge($alertas, $alertas_tiendas);
    
    // Verificar última actualización general
    $alertas_actualizacion = verificarUltimaActualizacion($pdo);
    $alertas = array_merge($alertas, $alertas_actualizacion);
    
    // Si hay alertas, enviar correo
    if (!empty($alertas)) {
        $asunto = "🚨 ALERTA: Stock sin actualizar o tiendas sin referencias - " . date('d/m/Y H:i');
        
        $mensaje = "<h2>Se han detectado las siguientes alertas:</h2>";
        
        foreach ($alertas as $alerta) {
            $color_clase = $alerta['tipo'] === 'CRITICA' ? 'alert' : 'warning';
            $icono = $alerta['tipo'] === 'CRITICA' ? '🚨' : '⚠️';
            
            $mensaje .= "
            <div class='$color_clase'>
                <h3>$icono {$alerta['tipo']}: {$alerta['mensaje']}</h3>
                <p><strong>Detalles:</strong> {$alerta['detalles']}</p>
                " . (isset($alerta['tienda']) ? "<p><strong>Tienda:</strong> {$alerta['tienda']}</p>" : "") . "
                " . (isset($alerta['seccion']) ? "<p><strong>Sección:</strong> {$alerta['seccion']}</p>" : "") . "
            </div>";
        }
        
        $mensaje .= "
        <div class='alert'>
            <h3>📋 Acciones Recomendadas:</h3>
            <ul>
                <li>Verificar que el proceso de importación de stock esté funcionando</li>
                <li>Revisar que el cron job de consolidación esté ejecutándose</li>
                <li>Contactar al equipo técnico si persisten los problemas</li>
            </ul>
        </div>";
        
        // Enviar correo
        $enviado = enviarCorreo($asunto, $mensaje, $config_correo);
        
        if ($enviado) {
            echo "✅ Alertas enviadas por correo exitosamente\n";
            echo "📧 Enviado a: " . implode(', ', $config_correo['to_emails']) . "\n";
        } else {
            echo "❌ Error enviando correo de alertas\n";
        }
        
        // Mostrar alertas en consola
        echo "\n🚨 ALERTAS DETECTADAS:\n";
        foreach ($alertas as $alerta) {
            echo "- {$alerta['tipo']}: {$alerta['mensaje']}\n";
        }
        
    } else {
        echo "✅ No se detectaron alertas. El stock está actualizado correctamente.\n";
    }
    
    return $alertas;
}

// Ejecutar si se llama directamente
if (php_sapi_name() === 'cli' || isset($_GET['ejecutar'])) {
    echo "🔍 Iniciando verificación de stock...\n";
    echo "📅 Fecha: " . date('d/m/Y H:i:s') . "\n\n";
    
    try {
        $alertas = ejecutarVerificaciones($pdo, $config_correo);
        
        echo "\n📊 RESUMEN:\n";
        echo "- Total alertas: " . count($alertas) . "\n";
        echo "- Alertas críticas: " . count(array_filter($alertas, fn($a) => $a['tipo'] === 'CRITICA')) . "\n";
        echo "- Alertas de advertencia: " . count(array_filter($alertas, fn($a) => $a['tipo'] === 'ADVERTENCIA')) . "\n";
        
    } catch (Exception $e) {
        echo "❌ Error durante la verificación: " . $e->getMessage() . "\n";
    }
    
    echo "\n✅ Verificación completada.\n";
}

// Función para verificación web (llamada desde interfaz)
function verificarStockWeb($pdo) {
    $alertas_tiendas = verificarStockPorTienda($pdo);
    $alertas_actualizacion = verificarUltimaActualizacion($pdo);
    
    return [
        'tiendas' => $alertas_tiendas,
        'actualizacion' => $alertas_actualizacion,
        'total_alertas' => count($alertas_tiendas) + count($alertas_actualizacion),
        'fecha_verificacion' => date('d/m/Y H:i:s')
    ];
}
?>
