<?php
/**
 * Script de Consolidación Automática de Stock
 * 
 * Este script se ejecuta automáticamente cada hora via cron job
 * para mantener actualizado el stock consolidado.
 * 
 * USO:
 * php ejecutar_consolidado.php
 * 
 * CRON JOB:
 * 0 * * * * /usr/bin/php /ruta/completa/a/ejecutar_consolidado.php
 */

// Configuración de zona horaria
date_default_timezone_set('America/Bogota');

// Configuración de logging
$log_file = __DIR__ . '/logs/consolidado_automatico.log';
$log_dir = dirname($log_file);

// Crear directorio de logs si no existe
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Función de logging
function logMessage($message, $type = 'INFO') {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[{$timestamp}] [{$type}] {$message}" . PHP_EOL;
    
    // Escribir en archivo de log
    file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
    
    // También mostrar en consola si se ejecuta manualmente
    if (php_sapi_name() === 'cli') {
        echo $log_entry;
    }
}

// Iniciar ejecución
logMessage('Iniciando consolidación automática de stock');

try {
    // Verificar si se puede conectar a la base de datos
    require_once '/home/crusardi/config_crusardi/database.php';
    
    if (!$pdo) {
        throw new Exception('No se pudo conectar a la base de datos');
    }
    
    logMessage('Conexión a base de datos establecida');
    
    // Configurar charset UTF-8
    $pdo->exec("SET NAMES 'utf8mb4'");
    $pdo->exec("SET CHARACTER SET utf8mb4");
    $pdo->exec("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");
    
    // Verificar que la tabla existe
    $check_table = $pdo->query("SHOW TABLES LIKE 'stock_calculado_cache'");
    if ($check_table->rowCount() == 0) {
        throw new Exception('La tabla stock_calculado_cache no existe');
    }
    
    logMessage('Tabla stock_calculado_cache verificada');
    
    // Ejecutar el procedimiento de consolidación
    logMessage('Ejecutando procedimiento consolidar_stock_crusardi');
    
    $start_time = microtime(true);
    
    $stmt = $pdo->prepare("CALL consolidar_stock_crusardi()");
    $stmt->execute();
    
    // Obtener resultados del procedimiento
    $resultados = [];
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows) {
            $resultados = array_merge($resultados, $rows);
        }
    } while ($stmt->nextRowset());
    
    $stmt->closeCursor();
    
    $end_time = microtime(true);
    $execution_time = round($end_time - $start_time, 2);
    
    logMessage("Procedimiento ejecutado en {$execution_time} segundos");
    
    // Obtener estadísticas actualizadas
    $stats = $pdo->query("
        SELECT 
            COUNT(DISTINCT codigo) as total_productos,
            SUM(stock_total) as total_unidades,
            MAX(ultima_actualizacion) as ultima_actualizacion
        FROM stock_calculado_cache 
        WHERE stock_total > 0
    ")->fetch(PDO::FETCH_ASSOC);
    
    // Log de estadísticas
    logMessage("Estadísticas actualizadas:");
    logMessage("- Total productos: " . number_format($stats['total_productos']));
    logMessage("- Total unidades: " . number_format($stats['total_unidades']));
    logMessage("- Última actualización: " . ($stats['ultima_actualizacion'] ?? 'N/A'));
    
    // Verificar si hay errores en los resultados
    $errors = array_filter($resultados, function($row) {
        return isset($row['error_detalle']) || 
               (isset($row['info']) && strpos($row['info'], 'Error') !== false);
    });
    
    if (!empty($errors)) {
        foreach ($errors as $error) {
            logMessage("Error detectado: " . json_encode($error), 'ERROR');
        }
        throw new Exception('Se detectaron errores durante la consolidación');
    }
    
    // Log de resultados exitosos
    foreach ($resultados as $resultado) {
        if (isset($resultado['info'])) {
            logMessage("Resultado: " . $resultado['info']);
        }
    }
    
    logMessage('Consolidación automática completada exitosamente');
    
    // Crear archivo de estado para monitoreo
    $status_file = __DIR__ . '/logs/consolidado_status.json';
    $status_data = [
        'ultima_ejecucion' => date('Y-m-d H:i:s'),
        'estado' => 'exitoso',
        'tiempo_ejecucion' => $execution_time,
        'total_productos' => $stats['total_productos'],
        'total_unidades' => $stats['total_unidades'],
        'errores' => count($errors)
    ];
    
    file_put_contents($status_file, json_encode($status_data, JSON_PRETTY_PRINT));
    
} catch (Exception $e) {
    logMessage('Error durante la consolidación: ' . $e->getMessage(), 'ERROR');
    
    // Crear archivo de estado con error
    $status_file = __DIR__ . '/logs/consolidado_status.json';
    $status_data = [
        'ultima_ejecucion' => date('Y-m-d H:i:s'),
        'estado' => 'error',
        'error' => $e->getMessage(),
        'errores' => 1
    ];
    
    file_put_contents($status_file, json_encode($status_data, JSON_PRETTY_PRINT));
    
    // En modo CLI, salir con código de error
    if (php_sapi_name() === 'cli') {
        exit(1);
    }
}

logMessage('Script de consolidación finalizado');
?>
