<?php
require_once '/home/crusardi/config_crusardi/database.php';
require_once __DIR__ . '/../../vendor/autoload.php';

header('Content-Type: text/plain; charset=utf-8');
@ob_end_clean();
set_time_limit(0);

function log_message($msg) {
    echo $msg . "\n";
    ob_flush();
    flush();
}

try {
    log_message("🚀 Iniciando proceso de importación de relaciones KIT...");
    log_message(" Fecha: " . date('d/m/Y H:i:s'));
    log_message("⏱️  Iniciando transacción...");

    $current_role = $_SESSION['user_role'] ?? 'viewer';
    if ($current_role === 'viewer') throw new Exception(" No tienes permisos para importar datos.");
    
    if (!isset($_FILES['relations_file']) || $_FILES['relations_file']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception("⚠️ Archivo no recibido o error en la subida.");
    }

    $filepath = $_FILES['relations_file']['tmp_name'];
    $filename = $_FILES['relations_file']['name'];
    
    log_message("📁 Archivo: " . $filename);
    log_message(" Tamaño: " . number_format(filesize($filepath)) . " bytes");
    
    $imported_count = 0;
    $updated_count = 0;
    $skipped_count = 0;
    $total_lines = 0;
    $imported_sample = [];
    $updated_sample = [];
    $skipped_sample = [];
    $sample_limit = 25;

    $pdo->beginTransaction();
    
    // =========================================================================
    // SOLUCIÓN INTELIGENTE: REPLACE - Actualiza si existe, inserta si es nuevo
    // =========================================================================
    log_message("🔄 Configurando REPLACE inteligente...");
    log_message("�� REPLACE: Actualiza relaciones existentes, inserta nuevas, mantiene el resto intacto");
    
    // Usar REPLACE que actualiza si existe, inserta si es nuevo
    $stmt = $pdo->prepare("REPLACE INTO lmat_relations (lmat, parte, qty_required, fecha_creacion) VALUES (?, ?, ?, NOW())");

    // Detección automática de formato (CSV o TXT)
    $fileHandle = fopen($filepath, 'r');
    if (!$fileHandle) {
        throw new Exception("ERROR: No se pudo abrir el archivo para lectura.");
    }
    
    $first_line = fgets($fileHandle);
    rewind($fileHandle); // Volver al inicio del archivo

    $first_line_clean = strtolower(str_replace([' ', "\t", "\n", "\r"], '', $first_line));
    $is_csv = (
        strpos($first_line_clean, 'lmat,parte,ctd_requerida') !== false ||
        strpos($first_line_clean, 'lmat,hijas,cantidad') !== false ||
        strpos($first_line_clean, 'lmat,parte,cantidad') !== false ||
        strpos($first_line_clean, 'kit,parte,cantidad') !== false
    );

    if ($is_csv) {
        log_message(" Formato CSV detectado. Procesando...");
        sleep(1);
        
        // Omitir cabecera
        $header = fgetcsv($fileHandle);
        log_message(" Cabecera detectada: " . implode(' | ', $header));
        $total_lines = 1;

        while (($data = fgetcsv($fileHandle)) !== FALSE) {
            $total_lines++;
            
            if (count($data) >= 3) {
                $lmat = strtoupper(trim($data[0]));
                $parte = strtoupper(trim($data[1]));
                $qty = intval(trim($data[2]));
                
                if ($lmat && $parte && $qty > 0) {
                    try {
                        // REPLACE: Si existe actualiza, si no existe inserta
                        $stmt->execute([$lmat, $parte, $qty]);
                        
                        // Verificar si fue INSERT o UPDATE
                        if ($stmt->rowCount() === 1) {
                            // Fue un INSERT nuevo
                            $imported_count++;
                            if (count($imported_sample) < $sample_limit) {
                                $imported_sample[] = "🆕 NUEVO: LMAT: <strong>$lmat</strong>, Parte: <strong>$parte</strong>, Cantidad: <strong>$qty</strong>";
                            }
                        } else {
                            // Fue un UPDATE (REPLACE)
                            $updated_count++;
                            if (count($updated_sample) < $sample_limit) {
                                $updated_sample[] = "�� ACTUALIZADO: LMAT: <strong>$lmat</strong>, Parte: <strong>$parte</strong>, Cantidad: <strong>$qty</strong>";
                            }
                        }
                        
                        // Mostrar progreso cada 100 registros
                        if (($imported_count + $updated_count) % 100 === 0) {
                            log_message(" Procesados " . ($imported_count + $updated_count) . " registros...");
                        }
                        
                    } catch (PDOException $e) {
                        $skipped_count++;
                        if (count($skipped_sample) < $sample_limit) {
                           $skipped_sample[] = "LMAT: <strong>$lmat</strong>, Parte: <strong>$parte</strong>, Cantidad: <strong>$qty</strong> (Error: " . $e->getMessage() . ")";
                        }
                    }
                } else {
                    $skipped_count++;
                    if (count($skipped_sample) < $sample_limit) {
                       $skipped_sample[] = "Línea $total_lines: Datos inválidos (LMAT: '$lmat', PARTE: '$parte', QTY: $qty)";
                    }
                }
            } else {
                $skipped_count++;
                if (count($skipped_sample) < $sample_limit) {
                   $skipped_sample[] = "Línea $total_lines: Formato incorrecto - " . implode(' | ', $data);
                }
            }
        }
        log_message("📊 Total de líneas procesadas: $total_lines");
        
    } else {
        log_message(" Formato de texto (Axapta) detectado. Procesando con lógica personalizada...");
        sleep(1);
        
        $current_lmat = null;
        $in_block = false;
        $block_count = 0;

        while (($line = fgets($fileHandle)) !== false) {
            $total_lines++;
            $line_normalized = trim(preg_replace('/\s+/', ' ', $line));

            if (stripos($line_normalized, 'Compra Inventario') !== false) {
                $in_block = true;
                $current_lmat = null;
                $block_count++;
                log_message("🔍 Bloque $block_count detectado");
                continue;
            }

            if ($in_block && !$current_lmat && preg_match('/^([A-Z0-9-]{8,}) \[.*?\].*?pcs.*?1[,\.]00/i', $line_normalized, $match)) {
                $current_lmat = $match[1];
                log_message("🟢 LMAT detectado: $current_lmat");
                continue;
            }

            if ($in_block && $current_lmat && preg_match('/^([A-Z0-9-]{8,}) \[.*?\].*? (\d{1,3}(?:[.,]\d{3})*)/', $line_normalized, $match)) {
                $parte = preg_replace('/\s+/', '', $match[1]);
                $qty_str = str_replace(['.', ','], ['', '.'], $match[2]);
                $qty = floatval($qty_str);
                
                if ($qty > 0) {
                    try {
                        // REPLACE: Si existe actualiza, si no existe inserta
                        $stmt->execute([$current_lmat, $parte, $qty]);
                        
                        // Verificar si fue INSERT o UPDATE
                        if ($stmt->rowCount() === 1) {
                            // Fue un INSERT nuevo
                            $imported_count++;
                            if (count($imported_sample) < $sample_limit) {
                                $imported_sample[] = "🆕 NUEVO: LMAT: <strong>$current_lmat</strong>, Parte: <strong>$parte</strong>, Cantidad: <strong>$qty</strong>";
                            }
                        } else {
                            // Fue un UPDATE (REPLACE)
                            $updated_count++;
                            if (count($updated_sample) < $sample_limit) {
                                $updated_sample[] = "�� ACTUALIZADO: LMAT: <strong>$current_lmat</strong>, Parte: <strong>$parte</strong>, Cantidad: <strong>$qty</strong>";
                            }
                        }
                        
                        // Mostrar progreso cada 50 registros
                        if (($imported_count + $updated_count) % 50 === 0) {
                            log_message(" Procesados " . ($imported_count + $updated_count) . " registros...");
                        }
                        
                    } catch (PDOException $e) {
                        $skipped_count++;
                        if (count($skipped_sample) < $sample_limit) {
                           $skipped_sample[] = "LMAT: <strong>$current_lmat</strong>, Parte: <strong>$parte</strong>, Cantidad: <strong>$qty</strong> (Error: " . $e->getMessage() . ")";
                        }
                    }
                }
            }
        }
        
        log_message("📊 Total de líneas procesadas: $total_lines");
        log_message("🔍 Total de bloques detectados: $block_count");
    }
    
    fclose($fileHandle);
    
    // =========================================================================
    // VALIDACIONES FINALES Y COMMIT
    // =========================================================================
    log_message("🔍 Validando resultados...");
    
    if (($imported_count + $updated_count) === 0) {
        throw new Exception("⚠️ No se procesó ninguna relación. Verifica el formato del archivo.");
    }
    
    // Verificar que se procesaron correctamente
    $verification = $pdo->query("SELECT COUNT(*) FROM lmat_relations")->fetchColumn();
    log_message("✅ Verificación: $verification relaciones totales en la base de datos");
    
    // Commit de la transacción
    $pdo->commit();
    log_message("💾 Transacción confirmada exitosamente");
    
    // Limpiar caché
    @unlink(__DIR__ . '/../../cache/public_catalog_data.json');
    log_message("🗑️  Caché del catálogo limpiado");

    // Guardar reporte en sesión
    $_SESSION['import_report'] = [
        'imported_count' => $imported_count,
        'updated_count' => $updated_count,
        'skipped_count' => $skipped_count,
        'total_lines' => $total_lines,
        'filename' => $filename,
        'imported' => $imported_sample, 
        'updated' => $updated_sample,
        'skipped' => $skipped_sample,
        'verification_count' => $verification
    ];

    // =========================================================================
    // REPORTE FINAL DETALLADO
    // =========================================================================
    log_message("\n" . str_repeat("=", 60));
    log_message(" PROCESO COMPLETADO EXITOSAMENTE");
    log_message(str_repeat("=", 60));
    log_message(" RESUMEN FINAL:");
    log_message("   �� Relaciones NUEVAS: $imported_count");
    log_message("   🔄 Relaciones ACTUALIZADAS: $updated_count");
    log_message("   Líneas omitidas/errores: $skipped_count");
    log_message("   📄 Total líneas procesadas: $total_lines");
    log_message("   Archivo: $filename");
    log_message("   ✅ Verificación BD: $verification registros totales");
    log_message("   🔄 Caché: Limpiado");
    log_message("");
    log_message("💡 IMPORTANTE: Solo se modificaron las relaciones del archivo");
    log_message("   Las demás relaciones en la BD se mantuvieron intactas.");
    log_message("");
    log_message("🔄 Haz clic en 'Cerrar y Recargar' para ver el resumen completo.");
    log_message(str_repeat("=", 60));

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
        log_message(" Transacción revertida debido al error");
    }
    
    log_message("\n" . str_repeat("❌", 20));
    log_message("ERROR CRÍTICO: " . $e->getMessage());
    log_message(str_repeat("❌", 20));
    log_message("");
    log_message("🔧 Soluciones posibles:");
    log_message("   1. Verifica el formato del archivo");
    log_message("   2. Asegúrate de que los códigos existan en references_data");
    log_message("   3. Revisa los permisos de la base de datos");
    log_message("   4. Contacta al administrador del sistema");
}
?>