<?php
// pages/descuentos.php

// Verificar permisos
if (!isset($_SESSION['user_role']) || !in_array($_SESSION['user_role'], ['admin', 'editor'])) {
    echo '<div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg">
            <p class="text-red-700">No tienes permisos para acceder a esta sección.</p>
          </div>';
    return;
}

// Crear tabla de descuentos si no existe
try {
    $checkTable = $pdo->query("SHOW TABLES LIKE 'descuentos'");
    if ($checkTable->rowCount() == 0) {
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS descuentos (
                id INT AUTO_INCREMENT PRIMARY KEY,
                tipo_descuento ENUM('general', 'categoria', 'referencia', 'referencia_ubicacion') NOT NULL,
                categoria_id INT NULL,
                referencia VARCHAR(100) NULL,
                tienda VARCHAR(50) NULL,
                bodega VARCHAR(50) NULL,
                porcentaje_descuento DECIMAL(5,2) NOT NULL,
                fecha_inicio DATE NULL,
                fecha_fin DATE NULL,
                descripcion TEXT NULL,
                activo TINYINT(1) DEFAULT 1,
                prioridad INT DEFAULT 0,
                created_by VARCHAR(100),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_tipo (tipo_descuento),
                INDEX idx_referencia (referencia),
                INDEX idx_categoria (categoria_id),
                INDEX idx_activo (activo),
                INDEX idx_fechas (fecha_inicio, fecha_fin)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
        ");
    }
} catch (PDOException $e) {
    echo '<div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg">
            <p class="text-red-700">Error al crear la tabla: ' . htmlspecialchars($e->getMessage()) . '</p>
          </div>';
    return;
}

// Procesar acciones POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $response = ['success' => false, 'message' => ''];
    
    try {
        if (isset($_POST['action'])) {
            switch ($_POST['action']) {
                case 'add':
                    // Validar datos según el tipo
                    $tipo = $_POST['tipo_descuento'];
                    $categoria_id = null;
                    $referencia = null;
                    $tienda = null;
                    $bodega = null;
                    
                    switch ($tipo) {
                        case 'categoria':
                            $categoria_id = $_POST['categoria_id'] ?: null;
                            break;
                        case 'referencia':
                            $referencia = $_POST['referencia'];
                            break;
                        case 'referencia_ubicacion':
                            $referencia = $_POST['referencia'];
                            $tienda = $_POST['tienda'] ?: null;
                            $bodega = $_POST['bodega'] ?: null;
                            break;
                    }
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO descuentos 
                        (tipo_descuento, categoria_id, referencia, tienda, bodega, 
                         porcentaje_descuento, fecha_inicio, fecha_fin, descripcion, 
                         prioridad, created_by)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    
                    $stmt->execute([
                        $tipo,
                        $categoria_id,
                        $referencia,
                        $tienda,
                        $bodega,
                        $_POST['porcentaje_descuento'],
                        !empty($_POST['fecha_inicio']) ? $_POST['fecha_inicio'] : null,
                        !empty($_POST['fecha_fin']) ? $_POST['fecha_fin'] : null,
                        $_POST['descripcion'] ?: null,
                        $_POST['prioridad'] ?: 0,
                        $_SESSION['user_email']
                    ]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Descuento agregado correctamente';
                    break;
                    
                case 'update':
                    $tipo = $_POST['tipo_descuento'];
                    $categoria_id = null;
                    $referencia = null;
                    $tienda = null;
                    $bodega = null;
                    
                    switch ($tipo) {
                        case 'categoria':
                            $categoria_id = $_POST['categoria_id'] ?: null;
                            break;
                        case 'referencia':
                            $referencia = $_POST['referencia'];
                            break;
                        case 'referencia_ubicacion':
                            $referencia = $_POST['referencia'];
                            $tienda = $_POST['tienda'] ?: null;
                            $bodega = $_POST['bodega'] ?: null;
                            break;
                    }
                    
                    $stmt = $pdo->prepare("
                        UPDATE descuentos 
                        SET tipo_descuento = ?, categoria_id = ?, referencia = ?, 
                            tienda = ?, bodega = ?, porcentaje_descuento = ?, 
                            fecha_inicio = ?, fecha_fin = ?, descripcion = ?, 
                            prioridad = ?, activo = ?
                        WHERE id = ?
                    ");
                    
                    $stmt->execute([
                        $tipo,
                        $categoria_id,
                        $referencia,
                        $tienda,
                        $bodega,
                        $_POST['porcentaje_descuento'],
                        !empty($_POST['fecha_inicio']) ? $_POST['fecha_inicio'] : null,
                        !empty($_POST['fecha_fin']) ? $_POST['fecha_fin'] : null,
                        $_POST['descripcion'] ?: null,
                        $_POST['prioridad'] ?: 0,
                        isset($_POST['activo']) ? 1 : 0,
                        $_POST['id']
                    ]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Descuento actualizado correctamente';
                    break;
                    
                case 'delete':
                    $stmt = $pdo->prepare("DELETE FROM descuentos WHERE id = ?");
                    $stmt->execute([$_POST['id']]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Descuento eliminado correctamente';
                    break;
                    
                case 'toggle':
                    $stmt = $pdo->prepare("UPDATE descuentos SET activo = NOT activo WHERE id = ?");
                    $stmt->execute([$_POST['id']]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Estado actualizado correctamente';
                    break;
            }
        }
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }
    
    // Si es petición AJAX, devolver JSON y terminar
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
        strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode($response);
        exit;
    }
}

// Obtener categorías para el selector
$categorias = [];
try {
    $stmt = $pdo->query("
        SELECT c1.id, c1.nombre as categoria1, c2.id as cat2_id, c2.nombre as categoria2 
        FROM categorias_nivel1 c1
        LEFT JOIN categorias_nivel2 c2 ON c1.id = c2.categoria1_id
        WHERE c1.activo = 1
        ORDER BY c1.orden, c1.nombre, c2.orden, c2.nombre
    ");
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        if (!isset($categorias[$row['id']])) {
            $categorias[$row['id']] = [
                'nombre' => $row['categoria1'],
                'subcategorias' => []
            ];
        }
        if ($row['cat2_id']) {
            $categorias[$row['id']]['subcategorias'][] = [
                'id' => $row['cat2_id'],
                'nombre' => $row['categoria2']
            ];
        }
    }
} catch (PDOException $e) {
    // Manejar error silenciosamente
}

// Obtener tiendas y bodegas únicas
$tiendas = [];
$bodegas = [];
try {
    $stmt = $pdo->query("SELECT DISTINCT bodega FROM stock_data ORDER BY bodega");
    $tiendas = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $stmt = $pdo->query("SELECT DISTINCT almacen FROM stock_data ORDER BY almacen");
    $bodegas = $stmt->fetchAll(PDO::FETCH_COLUMN);
} catch (PDOException $e) {
    // Manejar error silenciosamente
}

// Obtener filtros
$filtro_tipo = $_GET['tipo'] ?? 'todos';
$filtro_busqueda = $_GET['busqueda'] ?? '';
$filtro_activo = $_GET['activo'] ?? 'activos';

// Construir consulta con filtros
$query = "
    SELECT d.*, 
           c1.nombre as categoria_nombre,
           c2.nombre as subcategoria_nombre,
           r.nombre as referencia_nombre
    FROM descuentos d
    LEFT JOIN categorias_nivel1 c1 ON d.categoria_id = c1.id AND d.tipo_descuento = 'categoria'
    LEFT JOIN categorias_nivel2 c2 ON d.categoria_id = c2.id AND d.tipo_descuento = 'categoria'
    LEFT JOIN references_data r ON CONVERT(d.referencia USING utf8mb4) = CONVERT(r.referencia USING utf8mb4)
    WHERE 1=1
";
$params = [];

if ($filtro_tipo !== 'todos') {
    $query .= " AND d.tipo_descuento = ?";
    $params[] = $filtro_tipo;
}

if (!empty($filtro_busqueda)) {
    $query .= " AND (d.referencia LIKE ? OR d.descripcion LIKE ? OR r.nombre LIKE ?)";
    $busqueda_param = "%$filtro_busqueda%";
    $params = array_merge($params, [$busqueda_param, $busqueda_param, $busqueda_param]);
}

if ($filtro_activo === 'activos') {
    $query .= " AND d.activo = 1";
} elseif ($filtro_activo === 'inactivos') {
    $query .= " AND d.activo = 0";
}

$query .= " ORDER BY d.tipo_descuento, d.prioridad DESC, d.created_at DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$descuentos = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Obtener estadísticas
$stats = [
    'total' => count($descuentos),
    'activos' => 0,
    'general' => 0,
    'categoria' => 0,
    'referencia' => 0,
    'ubicacion' => 0
];

foreach ($descuentos as $desc) {
    if ($desc['activo']) $stats['activos']++;
    switch ($desc['tipo_descuento']) {
        case 'general': $stats['general']++; break;
        case 'categoria': $stats['categoria']++; break;
        case 'referencia': $stats['referencia']++; break;
        case 'referencia_ubicacion': $stats['ubicacion']++; break;
    }
}
?>

<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow-sm p-6">
        <div class="flex justify-between items-center">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">Gestión de Descuentos</h1>
                <p class="mt-1 text-sm text-gray-600">
                    Configura descuentos generales, por categoría, referencia o ubicación específica
                </p>
            </div>
            <button onclick="openModal('add')" 
                    class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                <i class="fas fa-plus mr-2"></i>Nuevo Descuento
            </button>
        </div>
    </div>

    <!-- Estadísticas -->
    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-4">
        <div class="bg-white rounded-lg shadow-sm p-4">
            <div class="flex items-center">
                <div class="p-2 bg-blue-100 rounded-lg">
                    <i class="fas fa-percent text-blue-600"></i>
                </div>
                <div class="ml-3">
                    <p class="text-xs text-gray-600">Total</p>
                    <p class="text-xl font-bold text-gray-900"><?= $stats['total'] ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-sm p-4">
            <div class="flex items-center">
                <div class="p-2 bg-green-100 rounded-lg">
                    <i class="fas fa-check-circle text-green-600"></i>
                </div>
                <div class="ml-3">
                    <p class="text-xs text-gray-600">Activos</p>
                    <p class="text-xl font-bold text-gray-900"><?= $stats['activos'] ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-sm p-4">
            <div class="flex items-center">
                <div class="p-2 bg-purple-100 rounded-lg">
                    <i class="fas fa-globe text-purple-600"></i>
                </div>
                <div class="ml-3">
                    <p class="text-xs text-gray-600">Generales</p>
                    <p class="text-xl font-bold text-gray-900"><?= $stats['general'] ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-sm p-4">
            <div class="flex items-center">
                <div class="p-2 bg-orange-100 rounded-lg">
                    <i class="fas fa-folder text-orange-600"></i>
                </div>
                <div class="ml-3">
                    <p class="text-xs text-gray-600">Categoría</p>
                    <p class="text-xl font-bold text-gray-900"><?= $stats['categoria'] ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-sm p-4">
            <div class="flex items-center">
                <div class="p-2 bg-indigo-100 rounded-lg">
                    <i class="fas fa-tag text-indigo-600"></i>
                </div>
                <div class="ml-3">
                    <p class="text-xs text-gray-600">Referencia</p>
                    <p class="text-xl font-bold text-gray-900"><?= $stats['referencia'] ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-sm p-4">
            <div class="flex items-center">
                <div class="p-2 bg-red-100 rounded-lg">
                    <i class="fas fa-map-marker-alt text-red-600"></i>
                </div>
                <div class="ml-3">
                    <p class="text-xs text-gray-600">Ubicación</p>
                    <p class="text-xl font-bold text-gray-900"><?= $stats['ubicacion'] ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Filtros -->
    <div class="bg-white rounded-lg shadow-sm p-6">
        <form method="GET" class="flex flex-wrap gap-4">
            <input type="hidden" name="page" value="descuentos">
            
            <div class="flex-1 min-w-[250px]">
                <label class="block text-sm font-medium text-gray-700 mb-2">Buscar</label>
                <input type="text" 
                       name="busqueda" 
                       value="<?= htmlspecialchars($filtro_busqueda) ?>"
                       placeholder="Referencia, descripción..."
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
            </div>
            
            <div class="w-48">
                <label class="block text-sm font-medium text-gray-700 mb-2">Tipo</label>
                <select name="tipo" 
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    <option value="todos">Todos</option>
                    <option value="general" <?= $filtro_tipo === 'general' ? 'selected' : '' ?>>General</option>
                    <option value="categoria" <?= $filtro_tipo === 'categoria' ? 'selected' : '' ?>>Por Categoría</option>
                    <option value="referencia" <?= $filtro_tipo === 'referencia' ? 'selected' : '' ?>>Por Referencia</option>
                    <option value="referencia_ubicacion" <?= $filtro_tipo === 'referencia_ubicacion' ? 'selected' : '' ?>>Por Ubicación</option>
                </select>
            </div>
            
            <div class="w-40">
                <label class="block text-sm font-medium text-gray-700 mb-2">Estado</label>
                <select name="activo" 
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    <option value="todos">Todos</option>
                    <option value="activos" <?= $filtro_activo === 'activos' ? 'selected' : '' ?>>Activos</option>
                    <option value="inactivos" <?= $filtro_activo === 'inactivos' ? 'selected' : '' ?>>Inactivos</option>
                </select>
            </div>
            
            <div class="flex items-end gap-2">
                <button type="submit" 
                        class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                    <i class="fas fa-search mr-2"></i>Filtrar
                </button>
                <a href="?page=descuentos" 
                   class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors">
                    <i class="fas fa-redo mr-2"></i>Limpiar
                </a>
            </div>
        </form>
    </div>

    <!-- Tabla de descuentos -->
    <div class="bg-white rounded-lg shadow-sm overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Tipo
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Aplicado a
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Descuento
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Vigencia
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Prioridad
                        </th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Estado
                        </th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Acciones
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php if (empty($descuentos)): ?>
                        <tr>
                            <td colspan="7" class="px-6 py-4 text-center text-gray-500">
                                No se encontraron descuentos
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($descuentos as $desc): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <?php
                                    $tipo_badge = [
                                        'general' => ['bg-purple-100 text-purple-800', 'fa-globe', 'General'],
                                        'categoria' => ['bg-orange-100 text-orange-800', 'fa-folder', 'Categoría'],
                                        'referencia' => ['bg-indigo-100 text-indigo-800', 'fa-tag', 'Referencia'],
                                        'referencia_ubicacion' => ['bg-red-100 text-red-800', 'fa-map-marker-alt', 'Ubicación']
                                    ];
                                    $badge = $tipo_badge[$desc['tipo_descuento']];
                                    ?>
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?= $badge[0] ?>">
                                        <i class="fas <?= $badge[1] ?> mr-1"></i><?= $badge[2] ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm">
                                        <?php switch ($desc['tipo_descuento']):
                                            case 'general': ?>
                                                <div class="text-gray-900 font-medium">Todos los productos</div>
                                            <?php break;
                                            case 'categoria': ?>
                                                <div class="text-gray-900 font-medium">
                                                    <?= htmlspecialchars($desc['categoria_nombre'] ?? $desc['subcategoria_nombre'] ?? 'Categoría #' . $desc['categoria_id']) ?>
                                                </div>
                                            <?php break;
                                            case 'referencia': ?>
                                                <div class="text-gray-900 font-medium"><?= htmlspecialchars($desc['referencia']) ?></div>
                                                <?php if ($desc['referencia_nombre']): ?>
                                                    <div class="text-gray-500 text-xs"><?= htmlspecialchars($desc['referencia_nombre']) ?></div>
                                                <?php endif; ?>
                                            <?php break;
                                            case 'referencia_ubicacion': ?>
                                                <div class="text-gray-900 font-medium"><?= htmlspecialchars($desc['referencia']) ?></div>
                                                <div class="text-gray-500 text-xs">
                                                    <?= $desc['tienda'] ? 'Tienda: ' . htmlspecialchars($desc['tienda']) : 'Todas las tiendas' ?>
                                                    <?= $desc['bodega'] ? ' | Bodega: ' . htmlspecialchars($desc['bodega']) : '' ?>
                                                </div>
                                            <?php break;
                                        endswitch; ?>
                                        <?php if ($desc['descripcion']): ?>
                                            <div class="text-gray-500 text-xs mt-1"><?= htmlspecialchars($desc['descripcion']) ?></div>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="text-lg font-semibold text-green-600">
                                        <?= number_format($desc['porcentaje_descuento'], 0) ?>%
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm">
                                    <?php
                                    $hoy = date('Y-m-d');
                                    $vigente = true;
                                    
                                    if ($desc['fecha_inicio'] && $desc['fecha_inicio'] > $hoy) {
                                        $vigente = false;
                                        echo '<span class="text-yellow-600"><i class="fas fa-clock mr-1"></i>Desde ' . date('d/m/Y', strtotime($desc['fecha_inicio'])) . '</span>';
                                    } elseif ($desc['fecha_fin'] && $desc['fecha_fin'] < $hoy) {
                                        $vigente = false;
                                        echo '<span class="text-red-600"><i class="fas fa-times-circle mr-1"></i>Expiró ' . date('d/m/Y', strtotime($desc['fecha_fin'])) . '</span>';
                                    } else {
                                        if ($desc['fecha_inicio'] || $desc['fecha_fin']) {
                                            echo '<div class="text-green-600"><i class="fas fa-check-circle mr-1"></i>Vigente</div>';
                                            echo '<div class="text-xs text-gray-500">';
                                            if ($desc['fecha_inicio']) echo 'Desde: ' . date('d/m/Y', strtotime($desc['fecha_inicio'])) . '<br>';
                                            if ($desc['fecha_fin']) echo 'Hasta: ' . date('d/m/Y', strtotime($desc['fecha_fin']));
                                            echo '</div>';
                                        } else {
                                            echo '<span class="text-gray-500">Sin restricción</span>';
                                        }
                                    }
                                    ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-center">
                                    <span class="text-sm font-medium text-gray-700"><?= $desc['prioridad'] ?></span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <button onclick="toggleStatus(<?= $desc['id'] ?>)" 
                                            class="relative inline-flex items-center h-6 rounded-full w-11 transition-colors
                                            <?= $desc['activo'] ? 'bg-green-500' : 'bg-gray-200' ?>">
                                        <span class="inline-block w-4 h-4 transform transition-transform bg-white rounded-full shadow
                                            <?= $desc['activo'] ? 'translate-x-6' : 'translate-x-1' ?>">
                                        </span>
                                    </button>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                    <button onclick='openModal("edit", <?= json_encode($desc) ?>)' 
                                            class="text-indigo-600 hover:text-indigo-900 mr-3">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button onclick="deleteDiscount(<?= $desc['id'] ?>)" 
                                            class="text-red-600 hover:text-red-900">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Información sobre prioridades -->
    <div class="bg-blue-50 border-l-4 border-blue-400 p-4">
        <div class="flex">
            <div class="flex-shrink-0">
                <i class="fas fa-info-circle text-blue-400"></i>
            </div>
            <div class="ml-3">
                <h3 class="text-sm font-medium text-blue-800">Sistema de Prioridades</h3>
                <div class="mt-2 text-sm text-blue-700">
                    <p>Los descuentos se aplican según el siguiente orden de prioridad:</p>
                    <ol class="list-decimal list-inside mt-1">
                        <li><strong>Referencia + Ubicación específica</strong> (mayor prioridad)</li>
                        <li><strong>Referencia específica</strong></li>
                        <li><strong>Categoría</strong></li>
                        <li><strong>Descuento general</strong> (menor prioridad)</li>
                    </ol>
                    <p class="mt-2">Cuando hay múltiples descuentos aplicables, se usa el de mayor prioridad. Entre descuentos del mismo tipo, se usa el valor de prioridad más alto.</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal para agregar/editar -->
<div id="discountModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <form id="discountForm" class="p-6">
                <input type="hidden" name="action" id="modalAction">
                <input type="hidden" name="id" id="discountId">
                
                <h2 class="text-xl font-bold text-gray-900 mb-4" id="modalTitle">Nuevo Descuento</h2>
                
                <div class="space-y-4">
                    <!-- Tipo de descuento -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Tipo de Descuento <span class="text-red-500">*</span>
                        </label>
                        <select name="tipo_descuento" 
                                id="tipo_descuento"
                                onchange="toggleFieldsByType()"
                                required
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                            <option value="">Seleccionar tipo...</option>
                            <option value="general">Descuento General</option>
                            <option value="categoria">Por Categoría</option>
                            <option value="referencia">Por Referencia</option>
                            <option value="referencia_ubicacion">Por Referencia y Ubicación</option>
                        </select>
                    </div>
                    
                    <!-- Campo de categoría -->
                    <div id="categoria_field" style="display: none;">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Categoría <span class="text-red-500">*</span>
                        </label>
                        <select name="categoria_id" 
                                id="categoria_id"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                            <option value="">Seleccionar categoría...</option>
                            <?php foreach ($categorias as $cat1_id => $cat1): ?>
                                <optgroup label="<?= htmlspecialchars($cat1['nombre']) ?>">
                                    <option value="<?= $cat1_id ?>"><?= htmlspecialchars($cat1['nombre']) ?> (Toda la categoría)</option>
                                    <?php foreach ($cat1['subcategorias'] as $cat2): ?>
                                        <option value="<?= $cat2['id'] ?>">— <?= htmlspecialchars($cat2['nombre']) ?></option>
                                    <?php endforeach; ?>
                                </optgroup>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Campo de referencia -->
                    <div id="referencia_field" style="display: none;">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Referencia <span class="text-red-500">*</span>
                        </label>
                        <input type="text" 
                               name="referencia" 
                               id="referencia"
                               placeholder="Código de referencia"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    </div>
                    
                    <!-- Campos de ubicación -->
                    <div id="ubicacion_fields" style="display: none;">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">
                                    Tienda/Bodega (Opcional)
                                </label>
                                <select name="tienda" 
                                        id="tienda"
                                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                                    <option value="">Todas las tiendas</option>
                                    <?php foreach ($tiendas as $tienda): ?>
                                        <option value="<?= htmlspecialchars($tienda) ?>"><?= htmlspecialchars($tienda) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">
                                    Almacén (Opcional)
                                </label>
                                <select name="bodega" 
                                        id="bodega"
                                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                                    <option value="">Todos los almacenes</option>
                                    <?php foreach ($bodegas as $bodega): ?>
                                        <option value="<?= htmlspecialchars($bodega) ?>"><?= htmlspecialchars($bodega) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Porcentaje de descuento -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Porcentaje de Descuento <span class="text-red-500">*</span>
                        </label>
                        <div class="relative">
                            <input type="number" 
                                   name="porcentaje_descuento" 
                                   id="porcentaje_descuento"
                                   min="0"
                                   max="100"
                                   step="0.01"
                                   required
                                   class="w-full px-3 py-2 pr-8 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                            <span class="absolute right-3 top-2 text-gray-500">%</span>
                        </div>
                    </div>
                    
                    <!-- Fechas de vigencia -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Fecha de Inicio (Opcional)
                            </label>
                            <input type="date" 
                                   name="fecha_inicio" 
                                   id="fecha_inicio"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Fecha de Fin (Opcional)
                            </label>
                            <input type="date" 
                                   name="fecha_fin" 
                                   id="fecha_fin"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                    </div>
                    
                    <!-- Prioridad y descripción -->
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Prioridad
                            </label>
                            <input type="number" 
                                   name="prioridad" 
                                   id="prioridad"
                                   min="0"
                                   value="0"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                        
                        <div class="md:col-span-2">
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                Descripción (Opcional)
                            </label>
                            <input type="text" 
                                   name="descripcion" 
                                   id="descripcion"
                                   placeholder="Descripción del descuento"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                    </div>
                    
                    <!-- Estado activo -->
                    <div class="flex items-center" id="activoContainer" style="display: none;">
                        <input type="checkbox" 
                               name="activo" 
                               id="activo"
                               checked
                               class="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300 rounded">
                        <label for="activo" class="ml-2 block text-sm text-gray-700">
                            Descuento activo
                        </label>
                    </div>
                </div>
                
                <div class="mt-6 flex justify-end space-x-3">
                    <button type="button" 
                            onclick="closeModal()"
                            class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors">
                        Cancelar
                    </button>
                    <button type="submit" 
                            class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                        <i class="fas fa-save mr-2"></i>Guardar
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Función para mostrar/ocultar campos según el tipo
function toggleFieldsByType() {
    const tipo = document.getElementById('tipo_descuento').value;
    
    // Ocultar todos los campos específicos
    document.getElementById('categoria_field').style.display = 'none';
    document.getElementById('referencia_field').style.display = 'none';
    document.getElementById('ubicacion_fields').style.display = 'none';
    
    // Limpiar required
    document.getElementById('categoria_id').removeAttribute('required');
    document.getElementById('referencia').removeAttribute('required');
    
    // Mostrar campos según el tipo
    switch (tipo) {
        case 'categoria':
            document.getElementById('categoria_field').style.display = 'block';
            document.getElementById('categoria_id').setAttribute('required', 'required');
            break;
        case 'referencia':
            document.getElementById('referencia_field').style.display = 'block';
            document.getElementById('referencia').setAttribute('required', 'required');
            break;
        case 'referencia_ubicacion':
            document.getElementById('referencia_field').style.display = 'block';
            document.getElementById('ubicacion_fields').style.display = 'block';
            document.getElementById('referencia').setAttribute('required', 'required');
            break;
    }
}

// Funciones del modal
function openModal(action, data = null) {
    const modal = document.getElementById('discountModal');
    const form = document.getElementById('discountForm');
    const title = document.getElementById('modalTitle');
    const actionInput = document.getElementById('modalAction');
    const activoContainer = document.getElementById('activoContainer');
    
    // Resetear formulario
    form.reset();
    
    // Configurar según la acción
    if (action === 'add') {
        title.textContent = 'Nuevo Descuento';
        actionInput.value = 'add';
        activoContainer.style.display = 'none';
        document.getElementById('activo').checked = true;
    } else if (action === 'edit' && data) {
        title.textContent = 'Editar Descuento';
        actionInput.value = 'update';
        activoContainer.style.display = 'flex';
        
        // Llenar campos
        document.getElementById('discountId').value = data.id;
        document.getElementById('tipo_descuento').value = data.tipo_descuento;
        toggleFieldsByType(); // Mostrar campos correctos
        
        if (data.categoria_id) {
            document.getElementById('categoria_id').value = data.categoria_id;
        }
        if (data.referencia) {
            document.getElementById('referencia').value = data.referencia;
        }
        if (data.tienda) {
            document.getElementById('tienda').value = data.tienda;
        }
        if (data.bodega) {
            document.getElementById('bodega').value = data.bodega;
        }
        
        document.getElementById('porcentaje_descuento').value = data.porcentaje_descuento;
        document.getElementById('fecha_inicio').value = data.fecha_inicio || '';
        document.getElementById('fecha_fin').value = data.fecha_fin || '';
        document.getElementById('prioridad').value = data.prioridad || 0;
        document.getElementById('descripcion').value = data.descripcion || '';
        document.getElementById('activo').checked = data.activo == 1;
    }
    
    modal.classList.remove('hidden');
}

function closeModal() {
    document.getElementById('discountModal').classList.add('hidden');
}

// Manejar envío del formulario
document.getElementById('discountForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    fetch('?page=descuentos', {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.message || 'Error al procesar la solicitud');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error al procesar la solicitud');
    });
});

// Función para cambiar estado
function toggleStatus(id) {
    if (!confirm('¿Desea cambiar el estado de este descuento?')) return;
    
    const formData = new FormData();
    formData.append('action', 'toggle');
    formData.append('id', id);
    
    fetch('?page=descuentos', {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.message || 'Error al cambiar el estado');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error al cambiar el estado');
    });
}

// Función para eliminar
function deleteDiscount(id) {
    if (!confirm('¿Está seguro de eliminar este descuento?')) return;
    
    const formData = new FormData();
    formData.append('action', 'delete');
    formData.append('id', id);
    
    fetch('?page=descuentos', {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.message || 'Error al eliminar');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error al eliminar');
    });
}

// Cerrar modal al hacer clic fuera
document.getElementById('discountModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeModal();
    }
});

// Validación de fechas
document.getElementById('fecha_inicio').addEventListener('change', function() {
    const inicio = this.value;
    const fin = document.getElementById('fecha_fin').value;
    
    if (inicio && fin && inicio > fin) {
        alert('La fecha de inicio no puede ser posterior a la fecha de fin');
        this.value = '';
    }
});

document.getElementById('fecha_fin').addEventListener('change', function() {
    const fin = this.value;
    const inicio = document.getElementById('fecha_inicio').value;
    
    if (inicio && fin && inicio > fin) {
        alert('La fecha de fin no puede ser anterior a la fecha de inicio');
        this.value = '';
    }
});
</script>