<?php
// pages/productos_imperfectos.php - VERSIÓN CORREGIDA

// Verificar permisos
if (!isset($_SESSION['user_role']) || !in_array($_SESSION['user_role'], ['admin', 'editor'])) {
    echo '<div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg">
            <p class="text-red-700">No tienes permisos para acceder a esta sección.</p>
          </div>';
    return;
}

// Definir las opciones de tienda y ubicación
$tiendas_disponibles = ['BW1', 'MD1', 'BB1', 'BB2'];
$ubicaciones_disponibles = ['SHOW', 'OWN', 'SAT', 'OUTLET'];

// Obtener categorías desde la base de datos
$categorias1 = [];
$categorias2 = [];
try {
    $stmt = $pdo->query("SELECT id, nombre FROM categorias_nivel1 WHERE activo = 1 ORDER BY orden, nombre");
    $categorias1 = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $stmt = $pdo->query("SELECT id, categoria1_id, nombre FROM categorias_nivel2 WHERE activo = 1 ORDER BY orden, nombre");
    $categorias2 = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Si no existen las tablas, usar arrays vacíos
}

// IMPORTANTE: Procesar formulario ANTES de cualquier output
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    try {
        switch ($_POST['action']) {
            case 'add':
                if (empty($_POST['codigo_unico'])) {
                    throw new Exception('El código único es obligatorio');
                }
                
                $stmt = $pdo->prepare("
                    INSERT INTO productos_imperfectos (
                        codigo_unico, lmat, referencias_hijas, nombre_lmat, nombre_hijas,
                        cantidad, precio, descuento, notas, created_by,
                        tienda, ubicacion, categoria1_id, categoria2_id
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $_POST['codigo_unico'],
                    $_POST['lmat'] ?: null,
                    $_POST['referencias_hijas'] ?: null,
                    $_POST['nombre_lmat'] ?: null,
                    $_POST['nombre_hijas'] ?: null,
                    intval($_POST['cantidad'] ?: 1),
                    floatval($_POST['precio'] ?: 0),
                    floatval($_POST['descuento'] ?: 0),
                    $_POST['notas'] ?: null,
                    $_SESSION['user_email'] ?? 'sistema',
                    $_POST['tienda'] ?: null,
                    $_POST['ubicacion'] ?: null,
                    $_POST['categoria1_id'] ?: null,
                    $_POST['categoria2_id'] ?: null
                ]);
                
                $_SESSION['mensaje_exito'] = 'Producto imperfecto agregado correctamente';
                break;
                
            case 'edit':
                if (empty($_POST['id'])) {
                    throw new Exception('ID de producto no recibido');
                }
                
                $stmt = $pdo->prepare("
                    UPDATE productos_imperfectos SET 
                        codigo_unico = ?, lmat = ?, referencias_hijas = ?,
                        nombre_lmat = ?, nombre_hijas = ?, cantidad = ?,
                        precio = ?, descuento = ?, notas = ?,
                        tienda = ?, ubicacion = ?, categoria1_id = ?, categoria2_id = ?
                    WHERE id = ?
                ");
                
                $stmt->execute([
                    $_POST['codigo_unico'],
                    $_POST['lmat'] ?: null,
                    $_POST['referencias_hijas'] ?: null,
                    $_POST['nombre_lmat'] ?: null,
                    $_POST['nombre_hijas'] ?: null,
                    intval($_POST['cantidad'] ?: 1),
                    floatval($_POST['precio'] ?: 0),
                    floatval($_POST['descuento'] ?: 0),
                    $_POST['notas'] ?: null,
                    $_POST['tienda'] ?: null,
                    $_POST['ubicacion'] ?: null,
                    $_POST['categoria1_id'] ?: null,
                    $_POST['categoria2_id'] ?: null,
                    intval($_POST['id'])
                ]);
                
                $_SESSION['mensaje_exito'] = 'Producto actualizado correctamente';
                break;
                
            case 'delete':
                if (empty($_POST['id'])) {
                    throw new Exception('ID de producto no recibido');
                }
                
                $stmt = $pdo->prepare("DELETE FROM productos_imperfectos WHERE id = ?");
                $stmt->execute([intval($_POST['id'])]);
                
                $_SESSION['mensaje_exito'] = 'Producto eliminado correctamente';
                break;
                
            default:
                throw new Exception('Acción no válida');
        }
        
    } catch (Exception $e) {
        $_SESSION['mensaje_error'] = 'Error: ' . $e->getMessage();
    }
    
    // Redirigir usando JavaScript porque el header ya fue enviado
    echo '<script>window.location.href = "index.php?page=productos_imperfectos";</script>';
    exit;
}

// Mostrar mensajes de sesión si existen
$mensaje_exito = '';
$mensaje_error = '';

if (isset($_SESSION['mensaje_exito'])) {
    $mensaje_exito = $_SESSION['mensaje_exito'];
    unset($_SESSION['mensaje_exito']);
}
if (isset($_SESSION['mensaje_error'])) {
    $mensaje_error = $_SESSION['mensaje_error'];
    unset($_SESSION['mensaje_error']);
}

// Obtener productos
$filtro_busqueda = $_GET['busqueda'] ?? '';
$query = "SELECT p.*, c1.nombre as cat1_nombre, c2.nombre as cat2_nombre 
          FROM productos_imperfectos p
          LEFT JOIN categorias_nivel1 c1 ON p.categoria1_id = c1.id
          LEFT JOIN categorias_nivel2 c2 ON p.categoria2_id = c2.id
          WHERE 1=1";
$params = [];

if (!empty($filtro_busqueda)) {
    $query .= " AND (p.codigo_unico LIKE ? OR p.lmat LIKE ? OR p.referencias_hijas LIKE ? 
                     OR p.nombre_lmat LIKE ? OR p.nombre_hijas LIKE ?)";
    $busqueda_param = "%$filtro_busqueda%";
    $params = array_merge($params, [$busqueda_param, $busqueda_param, $busqueda_param, 
                                    $busqueda_param, $busqueda_param]);
}

$query .= " ORDER BY p.created_at DESC";

try {
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $productos = [];
    $mensaje_error = 'Error al obtener productos: ' . $e->getMessage();
}

// Estadísticas
$total_productos = count($productos);
$total_cantidad = array_sum(array_column($productos, 'cantidad'));

// Calcular productos con imagen
$con_imagen = 0;
foreach ($productos as $p) {
    if (!empty($p['lmat'])) {
        $con_imagen++;
    }
}
?>

<div class="space-y-6">
    <!-- Mensajes -->
    <?php if ($mensaje_exito): ?>
    <div class="bg-green-50 border-l-4 border-green-500 p-4 rounded-lg">
        <p class="text-green-700"><?= htmlspecialchars($mensaje_exito) ?></p>
    </div>
    <?php endif; ?>
    
    <?php if ($mensaje_error): ?>
    <div class="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg">
        <p class="text-red-700"><?= htmlspecialchars($mensaje_error) ?></p>
    </div>
    <?php endif; ?>

    <!-- Header -->
    <div class="bg-white rounded-lg shadow-sm p-6">
        <div class="flex justify-between items-center">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">Productos Imperfectos</h1>
                <p class="mt-1 text-sm text-gray-600">
                    Gestión de productos con imperfectos o en liquidación
                </p>
            </div>
            <div class="flex space-x-3">
                <button onclick="openModal('add')" 
                        class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                    <i class="fas fa-plus mr-2"></i>Nuevo Producto
                </button>
            </div>
        </div>
        
        <!-- Estadísticas -->
        <div class="mt-4 grid grid-cols-3 gap-4">
            <div class="bg-blue-50 rounded-lg p-3">
                <div class="text-sm text-blue-600">Total Productos</div>
                <div class="text-xl font-bold text-blue-900"><?= $total_productos ?></div>
            </div>
            <div class="bg-green-50 rounded-lg p-3">
                <div class="text-sm text-green-600">Total Unidades</div>
                <div class="text-xl font-bold text-green-900"><?= $total_cantidad ?></div>
            </div>
            <div class="bg-purple-50 rounded-lg p-3">
                <div class="text-sm text-purple-600">Con Imagen</div>
                <div class="text-xl font-bold text-purple-900"><?= $con_imagen ?></div>
            </div>
        </div>
    </div>

    <!-- Buscador -->
    <div class="bg-white rounded-lg shadow-sm p-4">
        <form method="GET" action="index.php" class="flex gap-3">
            <input type="hidden" name="page" value="productos_imperfectos">
            <div class="flex-1">
                <input type="text" 
                       name="busqueda" 
                       placeholder="Buscar por código único, LMAT o nombre..."
                       value="<?= htmlspecialchars($filtro_busqueda) ?>"
                       class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500">
            </div>
            <button type="submit" 
                    class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                <i class="fas fa-search mr-2"></i>Buscar
            </button>
            <a href="index.php?page=productos_imperfectos" 
               class="px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 transition-colors">
                <i class="fas fa-redo mr-2"></i>Limpiar
            </a>
        </form>
    </div>

    <!-- Tabla de productos -->
    <div class="bg-white rounded-lg shadow-sm overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-3 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-32">
                            Producto
                        </th>
                        <th class="px-3 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-20">
                            Imagen
                        </th>
                        <th class="px-3 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-24">
                            L.MAT
                        </th>
                        <th class="px-3 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-32">
                            Descripción
                        </th>
                        <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider w-20">
                            Precio
                        </th>
                        <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider w-16">
                            Cant.
                        </th>
                        <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider w-16">
                            Tienda
                        </th>
                        <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider w-20">
                            Ubicación
                        </th>
                        <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider w-32">
                            Categoría
                        </th>
                        <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider w-28">
                            Fotos
                        </th>
                        <th class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider w-20">
                            Acciones
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php if (empty($productos)): ?>
                        <tr>
                            <td colspan="11" class="px-6 py-8 text-center text-gray-500">
                                <div class="flex flex-col items-center">
                                    <i class="fas fa-box-open text-4xl text-gray-300 mb-3"></i>
                                    <p class="text-lg font-medium">No se encontraron productos imperfectos</p>
                                    <p class="text-sm text-gray-400 mt-1">Intenta con otros criterios de búsqueda</p>
                                </div>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($productos as $producto): ?>
                        <tr class="hover:bg-gray-50">
                            <!-- Columna Producto -->
                            <td class="px-3 py-4">
                                <div class="text-sm font-medium text-gray-900">
                                    <?= htmlspecialchars($producto['codigo_unico']) ?>
                                </div>
                                <?php if ($producto['referencias_hijas']): ?>
                                    <div class="text-xs text-gray-500">
                                        Hijas: <?= htmlspecialchars(substr($producto['referencias_hijas'], 0, 30)) ?>...
                                    </div>
                                <?php endif; ?>
                            </td>

                            <!-- Columna Imagen -->
                            <td class="px-3 py-4">
                                <?php if (!empty($producto['lmat'])): ?>
                                    <?php $img_url = "https://imagenes.crusardi.net/boconcept/STOCKCOMERCIAL/" . $producto['lmat'] . ".jpg"; ?>
                                    <img src="<?= $img_url ?>" 
                                         alt="<?= htmlspecialchars($producto['lmat']) ?>"
                                         class="w-12 h-12 object-cover rounded"
                                         onerror="this.onerror=null; this.src='https://imagenes.crusardi.net/boconcept/sin_foto.png';">
                                <?php else: ?>
                                    <div class="w-12 h-12 bg-gray-200 rounded flex items-center justify-center">
                                        <i class="fas fa-image text-gray-400"></i>
                                    </div>
                                <?php endif; ?>
                            </td>

                            <!-- Columna L.MAT -->
                            <td class="px-3 py-4">
                                <div class="text-sm text-gray-900">
                                    <?= htmlspecialchars($producto['lmat'] ?: '-') ?>
                                </div>
                            </td>

                            <!-- Columna Descripción -->
                            <td class="px-3 py-4">
                                <div class="text-sm text-gray-900">
                                    <?= htmlspecialchars($producto['nombre_lmat'] ?: 'Sin nombre') ?>
                                </div>
                                <?php if ($producto['nombre_hijas']): ?>
                                    <div class="text-xs text-gray-500 mt-1">
                                        <?= htmlspecialchars(substr($producto['nombre_hijas'], 0, 50)) ?>...
                                    </div>
                                <?php endif; ?>
                            </td>

                            <!-- Columna Precio -->
                            <td class="px-3 py-4 text-center">
                                <div class="text-sm font-medium text-gray-900">
                                    $<?= number_format($producto['precio'], 0) ?>
                                </div>
                                <?php if ($producto['descuento'] > 0): ?>
                                    <div class="text-xs text-red-600">
                                        -<?= $producto['descuento'] ?>%
                                    </div>
                                <?php endif; ?>
                            </td>

                            <!-- Columna Cantidad -->
                            <td class="px-3 py-4 text-center">
                                <span class="text-sm font-medium text-gray-900">
                                    <?= $producto['cantidad'] ?>
                                </span>
                            </td>
                            
                            <!-- Columna Tienda -->
                            <td class="px-3 py-2 text-center">
                                <?php if ($producto['tienda']): ?>
                                    <span class="px-2 py-1 text-xs rounded-full bg-blue-100 text-blue-700 font-medium">
                                        <?= htmlspecialchars($producto['tienda']) ?>
                                    </span>
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                            
                            <!-- Columna Ubicación -->
                            <td class="px-3 py-2 text-center">
                                <?php if ($producto['ubicacion']): ?>
                                    <span class="px-2 py-1 text-xs rounded-full bg-green-100 text-green-700 font-medium">
                                        <?= htmlspecialchars($producto['ubicacion']) ?>
                                    </span>
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                            
                            <!-- Columna Categoría -->
                            <td class="px-3 py-2">
                                <?php if (!empty($producto['cat1_nombre'])): ?>
                                    <div class="text-xs">
                                        <div class="font-medium text-gray-700"><?= htmlspecialchars($producto['cat1_nombre']) ?></div>
                                        <?php if (!empty($producto['cat2_nombre'])): ?>
                                            <div class="text-gray-500">→ <?= htmlspecialchars($producto['cat2_nombre']) ?></div>
                                        <?php endif; ?>
                                    </div>
                                <?php else: ?>
                                    <span class="text-gray-400">-</span>
                                <?php endif; ?>
                            </td>

                            <!-- Columna Fotos - CORREGIDA LA RUTA DEL ACTION -->
                            <td class="px-3 py-4">
                                <!-- Formulario para subir fotos -->
                                <div class="flex items-center justify-center space-x-1">
                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                        <form action="pages/upload_foto_imperfectos.php" 
                                              method="POST" 
                                              enctype="multipart/form-data" 
                                              class="inline upload-form">
                                            <input type="hidden" name="codigo_unico" value="<?= htmlspecialchars($producto['codigo_unico']) ?>">
                                            <input type="hidden" name="foto_num" value="<?= $i ?>">
                                            <input type="file" 
                                                   name="foto" 
                                                   accept="image/*" 
                                                   class="hidden" 
                                                   id="foto_<?= $producto['id'] ?>_<?= $i ?>"
                                                   onchange="this.form.submit()">
                                            <button type="button" 
                                                    onclick="document.getElementById('foto_<?= $producto['id'] ?>_<?= $i ?>').click()"
                                                    class="w-6 h-6 text-xs rounded <?= !empty($producto['imagen_detalle_' . $i]) ? 'bg-blue-600' : 'bg-gray-500' ?> text-white hover:opacity-80"
                                                    title="Subir foto <?= $i ?>">
                                                <?= $i ?>
                                            </button>
                                        </form>
                                    <?php endfor; ?>
                                </div>
                                
                                <!-- Previsualización de fotos existentes -->
                                <div class="flex flex-wrap gap-1 mt-1">
                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                        <?php 
                                        $img_url = "https://imagenes.crusardi.net/boconcept/SALE2019/N " . 
                                                  urlencode($producto['codigo_unico']) . "/" . $i . ".jpg";
                                        ?>
                                        <div class="relative">
                                            <img src="<?= $img_url ?>" 
                                                 alt="Foto <?= $i ?>"
                                                 class="w-6 h-6 object-cover rounded border cursor-pointer hover:scale-125 transition-transform"
                                                 onclick="showImageModal('<?= htmlspecialchars($img_url) ?>')"
                                                 onerror="this.style.display='none'"
                                                 title="Foto <?= $i ?> - Click para ampliar"
                                                 loading="lazy">
                                        </div>
                                    <?php endfor; ?>
                                </div>
                            </td>

                            <!-- Columna Acciones -->
                            <td class="px-3 py-4">
                                <div class="flex items-center justify-center space-x-2">
                                    <button onclick='openModal("edit", <?= json_encode($producto) ?>)' 
                                            class="w-8 h-8 bg-indigo-100 text-indigo-600 rounded-lg hover:bg-indigo-200 transition-colors flex items-center justify-center"
                                            title="Editar producto">
                                        <i class="fas fa-edit text-sm"></i>
                                    </button>
                                    <form method="POST" class="inline" onsubmit="return confirm('¿Está seguro de eliminar este producto?');">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="id" value="<?= $producto['id'] ?>">
                                        <button type="submit"
                                                class="w-8 h-8 bg-red-100 text-red-600 rounded-lg hover:bg-red-200 transition-colors flex items-center justify-center"
                                                title="Eliminar producto">
                                            <i class="fas fa-trash text-sm"></i>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal para agregar/editar producto -->
<div id="productModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 max-w-4xl shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <h3 id="modalTitle" class="text-lg leading-6 font-medium text-gray-900 mb-4">
                Agregar Producto Imperfecto
            </h3>
            
            <form id="productForm" method="POST" action="index.php?page=productos_imperfectos" class="space-y-4">
                <input type="hidden" name="action" id="formAction" value="add">
                <input type="hidden" name="id" id="productId">
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Código Único <span class="text-red-500">*</span>
                        </label>
                        <input type="text" 
                               name="codigo_unico" 
                               id="codigo_unico" 
                               required
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            L.MAT
                        </label>
                        <input type="text" 
                               name="lmat" 
                               id="lmat"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    </div>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Nombre Producto
                    </label>
                    <input type="text" 
                           name="nombre_lmat" 
                           id="nombre_lmat"
                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Referencias Hijas
                    </label>
                    <div id="referencias_container">
                        <div class="flex gap-2 mb-2">
                            <input type="text" 
                                   name="referencias_hijas" 
                                   id="referencias_hijas"
                                   placeholder="Referencias separadas por comas"
                                   class="flex-1 px-3 py-2 border border-gray-300 rounded-lg">
                        </div>
                    </div>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Nombres Hijas
                    </label>
                    <div id="nombres_container">
                        <div class="flex gap-2 mb-2">
                            <input type="text" 
                                   name="nombre_hijas" 
                                   id="nombre_hijas"
                                   placeholder="Nombres separados por comas"
                                   class="flex-1 px-3 py-2 border border-gray-300 rounded-lg">
                        </div>
                    </div>
                </div>
                
                <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Cantidad
                        </label>
                        <input type="number" 
                               name="cantidad" 
                               id="cantidad" 
                               min="1" 
                               value="1"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Precio
                        </label>
                        <input type="number" name="precio" id="precio" step="0.01"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Descuento (%)
                        </label>
                        <input type="number" name="descuento" id="descuento" min="0" max="100" step="0.01"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                    </div>
                </div>
                
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Tienda
                        </label>
                        <select name="tienda" id="tienda" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                            <option value="">Sin asignar</option>
                            <?php foreach ($tiendas_disponibles as $tienda): ?>
                                <option value="<?= $tienda ?>"><?= $tienda ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Ubicación
                        </label>
                        <select name="ubicacion" id="ubicacion"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                            <option value="">Sin asignar</option>
                            <?php foreach ($ubicaciones_disponibles as $ubicacion): ?>
                                <option value="<?= $ubicacion ?>"><?= $ubicacion ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Categoría Principal
                        </label>
                        <select name="categoria1_id" id="categoria1_id" onchange="updateSubcategorias()"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                            <option value="">Sin categoría</option>
                            <?php foreach ($categorias1 as $cat): ?>
                                <option value="<?= $cat['id'] ?>"><?= htmlspecialchars($cat['nombre']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Subcategoría
                        </label>
                        <select name="categoria2_id" id="categoria2_id"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500">
                            <option value="">Sin subcategoría</option>
                        </select>
                    </div>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Notas
                    </label>
                    <textarea name="notas" 
                              id="notas"
                              rows="3"
                              class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500"></textarea>
                </div>
                
                <div class="mt-6 flex justify-end space-x-3">
                    <button type="button" 
                            onclick="closeModal()"
                            class="px-4 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors">
                        Cancelar
                    </button>
                    <button type="submit" 
                            class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                        <i class="fas fa-save mr-2"></i>Guardar
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal para ver imagen -->
<div id="imageModal" class="fixed inset-0 bg-black bg-opacity-90 hidden z-50" onclick="closeImageModal()">
    <div class="flex items-center justify-center min-h-screen p-4">
        <img id="modalImage" src="" alt="" class="max-w-full max-h-full">
    </div>
</div>

<script>
// Datos de categorías para JavaScript
const categorias2 = <?= json_encode($categorias2) ?>;

function openModal(action, product = null) {
    document.getElementById('productModal').classList.remove('hidden');
    document.getElementById('formAction').value = action;
    
    if (action === 'edit' && product) {
        document.getElementById('modalTitle').textContent = 'Editar Producto Imperfecto';
        document.getElementById('productId').value = product.id;
        document.getElementById('codigo_unico').value = product.codigo_unico;
        document.getElementById('lmat').value = product.lmat || '';
        document.getElementById('nombre_lmat').value = product.nombre_lmat || '';
        document.getElementById('referencias_hijas').value = product.referencias_hijas || '';
        document.getElementById('nombre_hijas').value = product.nombre_hijas || '';
        document.getElementById('cantidad').value = product.cantidad || 1;
        document.getElementById('precio').value = product.precio || 0;
        document.getElementById('descuento').value = product.descuento || 0;
        document.getElementById('notas').value = product.notas || '';
        document.getElementById('tienda').value = product.tienda || '';
        document.getElementById('ubicacion').value = product.ubicacion || '';
        document.getElementById('categoria1_id').value = product.categoria1_id || '';
        
        // Actualizar subcategorías si hay categoría seleccionada
        if (product.categoria1_id) {
            updateSubcategorias(product.categoria2_id);
        }
    } else {
        document.getElementById('modalTitle').textContent = 'Agregar Producto Imperfecto';
        document.getElementById('productForm').reset();
    }
}

function closeModal() {
    document.getElementById('productModal').classList.add('hidden');
}

function updateSubcategorias(selectedId = null) {
    const categoria1Id = document.getElementById('categoria1_id').value;
    const categoria2Select = document.getElementById('categoria2_id');
    
    // Limpiar opciones actuales
    categoria2Select.innerHTML = '<option value="">Sin subcategoría</option>';
    
    if (categoria1Id) {
        // Filtrar subcategorías por categoría principal
        const subcategorias = categorias2.filter(cat => cat.categoria1_id == categoria1Id);
        
        subcategorias.forEach(cat => {
            const option = document.createElement('option');
            option.value = cat.id;
            option.textContent = cat.nombre;
            if (selectedId && cat.id == selectedId) {
                option.selected = true;
            }
            categoria2Select.appendChild(option);
        });
    }
}

function showImageModal(imageSrc) {
    document.getElementById('modalImage').src = imageSrc;
    document.getElementById('imageModal').classList.remove('hidden');
}

function closeImageModal() {
    document.getElementById('imageModal').classList.add('hidden');
    document.getElementById('modalImage').src = '';
}

// Para debug si es necesario
function addReferencia() {
    console.log('Función para agregar referencia - implementar si es necesario');
}

// Envío automático de formularios de carga de fotos
document.addEventListener('DOMContentLoaded', function() {
    const uploadForms = document.querySelectorAll('.upload-form');
    uploadForms.forEach(form => {
        const fileInput = form.querySelector('input[type="file"]');
        if (fileInput) {
            fileInput.addEventListener('change', function() {
                if (this.files && this.files[0]) {
                    form.submit();
                }
            });
        }
    });
});
</script>