<?php
// pages/stock_calculado_functions.php

function generate_combinations($pdo) {
    // 1. Obtener recetas LMAT
    $lmat_recipes_from_db = [];
    $stmt_recipes = $pdo->query("SELECT lmat, parte, qty_required FROM lmat_relations");
    while ($row = $stmt_recipes->fetch()) {
        $lmat_recipes_from_db[$row['lmat']][] = ['parte' => $row['parte'], 'qty' => (int)$row['qty_required']];
    }

    // 2. Obtener datos de referencias con categorías
    $references_data = [];
    $stmt_refs = $pdo->query("SELECT * FROM references_data");
    while ($row = $stmt_refs->fetch()) {
        $references_data[$row['referencia']] = $row;
    }

    // 3. Obtener stock normal
    $stockByPartAndLocation = [];
    $stmt_stock = $pdo->query("SELECT referencia, bodega, almacen, cantidad FROM stock_data");
    $bodegasPermitidas = ['SHOW', 'OWN', 'SAT', 'OUTLET'];
    while ($row = $stmt_stock->fetch()) {
        if(in_array($row['almacen'], $bodegasPermitidas)) {
            $key = "{$row['referencia']}#{$row['bodega']}#{$row['almacen']}";
            $stockByPartAndLocation[$key] = (int)$row['cantidad'];
        }
    }

    // 4. Procesar excepciones de stock - MARCADAS COMO "PARTE"
    $exceptions_stmt = $pdo->query("SELECT * FROM stock_exceptions WHERE activo = 1");
    $price_overrides = [];
    $stock_exceptions_refs = []; // Referencias que son excepciones
    while ($ex = $exceptions_stmt->fetch()) {
        if (!empty($ex['cantidad_ajuste']) && !empty($ex['tienda']) && !empty($ex['ubicacion'])) {
            $key = "{$ex['referencia']}#{$ex['tienda']}#{$ex['ubicacion']}";
            $stockByPartAndLocation[$key] = ($stockByPartAndLocation[$key] ?? 0) + (int)$ex['cantidad_ajuste'];
            // Marcar esta referencia como excepción
            $stock_exceptions_refs[$ex['referencia']] = true;
        }
        if (!empty($ex['precio_override']) || !empty($ex['descuento_override'])) {
            $price_overrides[$ex['referencia']] = [
                'precio' => $ex['precio_override'], 
                'descuento' => $ex['descuento_override']
            ];
        }
    }

    // 5. Obtener excepciones de componentes de kit
    $kit_exceptions_stmt = $pdo->query("
        SELECT codigo_articulo, etiqueta_como, precio_individual 
        FROM kit_component_exceptions 
        WHERE activo = 1
    ");
    $kit_exceptions = [];
    while ($ke = $kit_exceptions_stmt->fetch()) {
        $kit_exceptions[$ke['codigo_articulo']] = [
            'etiqueta' => $ke['etiqueta_como'],
            'precio' => $ke['precio_individual']
        ];
    }

    // 6. Obtener productos imperfectos
    $imperfectos_stmt = $pdo->query("
        SELECT codigo_unico, lmat, referencias_hijas, nombre_lmat, nombre_hijas, 
               cantidad, precio, descuento, imagen_custom, notas,
               imagen_detalle_1, imagen_detalle_2, imagen_detalle_3, 
               imagen_detalle_4, imagen_detalle_5
        FROM productos_imperfectos 
        WHERE cantidad > 0
    ");
    $productos_imperfectos = [];
    while ($imp = $imperfectos_stmt->fetch()) {
        $productos_imperfectos[] = $imp;
    }

    // 7. Generar combinaciones
    $allCombinations = [];
    $uniqueLocations = [];
    
    foreach (array_keys($stockByPartAndLocation) as $key) {
        list($ref, $bodega, $almacen) = explode('#', $key);
        $location_key = "$bodega#$almacen";
        if (!in_array($location_key, $uniqueLocations)) $uniqueLocations[] = $location_key;
    }

    // Generar combinaciones para LMATs
    foreach ($lmat_recipes_from_db as $lmat => $recipe) {
        foreach ($uniqueLocations as $location) {
            $possibleKits = INF;
            list($bodega, $almacen) = explode('#', $location);
            $es_excepcion = false;
            
            foreach ($recipe as $part) {
                $stockKey = "{$part['parte']}#$bodega#$almacen";
                $availableStock = $stockByPartAndLocation[$stockKey] ?? 0;
                if ($availableStock < 0) $availableStock = 0;
                $kitsForThisPart = $part['qty'] > 0 ? floor($availableStock / $part['qty']) : INF;
                $possibleKits = min($possibleKits, $kitsForThisPart);
                
                // Verificar si alguna parte es una excepción
                if (isset($stock_exceptions_refs[$part['parte']])) {
                    $es_excepcion = true;
                }
            }
            
            if ($possibleKits > 0 && $possibleKits < INF) {
                $lmat_data = $references_data[$lmat] ?? null;
                $item = [
                    'id' => "$lmat#$bodega#$almacen",
                    'lmat' => $lmat,
                    'nombre' => $lmat_data['nombre'] ?? '',
                    'type' => 'lmat',
                    'units' => $possibleKits,
                    'bodega' => $bodega,
                    'almacen' => $almacen,
                    'precio' => $lmat_data['precio'] ?? 0,
                    'descuento' => 0,
                    'url_imagen' => $lmat_data['url_imagen'] ?? '',
                    'categoria' => $lmat_data['categoria'] ?? 'Sin categoría',
                    'categoria1' => $lmat_data['categoria1'] ?? 'Otros',
                    'categoria2' => $lmat_data['categoria2'] ?? 'General',
                    'es_excepcion' => $es_excepcion,
                    'etiqueta' => $es_excepcion ? 'Parte' : ''
                ];
                
                // Aplicar price overrides si existen
                if (isset($price_overrides[$lmat])) {
                    $item['precio'] = $price_overrides[$lmat]['precio'] ?? $item['precio'];
                    $item['descuento'] = $price_overrides[$lmat]['descuento'] ?? $item['descuento'];
                }
                
                $item['precio_final'] = $item['precio'] * (1 - ($item['descuento'] / 100.0));
                $allCombinations[] = $item;
            }
        }
    }

    // Agregar artículos individuales con stock
    foreach ($stockByPartAndLocation as $key => $stock) {
        if ($stock > 0) {
            list($ref, $bodega, $almacen) = explode('#', $key);
            $part_data = $references_data[$ref] ?? null;
            
            // Verificar si es una excepción de componente de kit
            $es_kit_exception = isset($kit_exceptions[$ref]);
            $etiqueta_articulo = '';
            $precio_override = null;
            
            if ($es_kit_exception) {
                $etiqueta_articulo = $kit_exceptions[$ref]['etiqueta'] ?? 'Parte';
                $precio_override = $kit_exceptions[$ref]['precio'];
            } elseif (isset($stock_exceptions_refs[$ref])) {
                $etiqueta_articulo = 'Parte';
            }
            
            $item = [
                'id' => "$ref#$bodega#$almacen#articulo",
                'lmat' => $ref,
                'nombre' => $part_data['nombre'] ?? '',
                'type' => 'articulo',
                'units' => $stock,
                'bodega' => $bodega,
                'almacen' => $almacen,
                'precio' => $precio_override ?? ($part_data['precio'] ?? 0),
                'descuento' => 0,
                'url_imagen' => $part_data['url_imagen'] ?? '',
                'categoria' => $part_data['categoria'] ?? 'Sin categoría',
                'categoria1' => $part_data['categoria1'] ?? 'Otros',
                'categoria2' => $part_data['categoria2'] ?? 'General',
                'es_excepcion' => $es_kit_exception || isset($stock_exceptions_refs[$ref]),
                'etiqueta' => $etiqueta_articulo
            ];
            
            // Aplicar price overrides de stock_exceptions
            if (isset($price_overrides[$ref])) {
                $item['precio'] = $price_overrides[$ref]['precio'] ?? $item['precio'];
                $item['descuento'] = $price_overrides[$ref]['descuento'] ?? $item['descuento'];
            }
            
            $item['precio_final'] = $item['precio'] * (1 - ($item['descuento'] / 100.0));
            $allCombinations[] = $item;
        }
    }

    // Agregar productos imperfectos como items especiales
    foreach ($productos_imperfectos as $imperfecto) {
        // Buscar datos de referencia del LMAT si existe
        $ref_data = isset($references_data[$imperfecto['lmat']]) ? 
                    $references_data[$imperfecto['lmat']] : null;
        
        // Determinar la imagen principal
        $imagen_principal = $imperfecto['imagen_custom'] ?? 
                           $imperfecto['imagen_detalle_1'] ?? 
                           ($ref_data['url_imagen'] ?? '');
        
        $item = [
            'id' => "IMP_{$imperfecto['codigo_unico']}",
            'lmat' => $imperfecto['lmat'] ?? $imperfecto['codigo_unico'],
            'nombre' => $imperfecto['nombre_lmat'] ?? 'Producto Imperfecto',
            'type' => 'imperfecto',
            'units' => $imperfecto['cantidad'],
            'bodega' => 'IMPERFECTOS',
            'almacen' => 'LIQUIDACION',
            'precio' => $imperfecto['precio'] ?? 0,
            'descuento' => $imperfecto['descuento'] ?? 0,
            'url_imagen' => $imagen_principal,
            'categoria' => $ref_data['categoria'] ?? 'Liquidación',
            'categoria1' => $ref_data['categoria1'] ?? 'Liquidación',
            'categoria2' => $ref_data['categoria2'] ?? 'Imperfectos',
            'es_imperfecto' => true,
            'codigo_unico' => $imperfecto['codigo_unico'],
            'etiqueta' => $imperfecto['codigo_unico'], // El código único como etiqueta
            'notas' => $imperfecto['notas'],
            'imagenes_detalle' => array_filter([
                $imperfecto['imagen_detalle_1'],
                $imperfecto['imagen_detalle_2'],
                $imperfecto['imagen_detalle_3'],
                $imperfecto['imagen_detalle_4'],
                $imperfecto['imagen_detalle_5']
            ])
        ];
        
        $item['precio_final'] = $item['precio'] * (1 - ($item['descuento'] / 100.0));
        $allCombinations[] = $item;
    }

    return ['combinations' => $allCombinations, 'recipes' => $lmat_recipes_from_db];
}

function filter_and_sort($combinations, $params) {
    if (!is_array($params)) {
        $params = [];
    }
    
    $search = $params['search'] ?? '';
    $show_type = $params['show_type'] ?? 'todos';
    $sort = $params['sort'] ?? 'alpha_asc';
    $bodegas = $params['bodegas'] ?? [];
    $almacenes = $params['almacenes'] ?? [];
    $qty = (int)($params['qty'] ?? 0);
    $qty_op = $params['qty_op'] ?? '>=';
    $precio = (float)($params['precio'] ?? 0);
    $precio_op = $params['precio_op'] ?? '>=';
    $descuento = (int)($params['descuento'] ?? 0);
    $descuento_op = $params['descuento_op'] ?? '>=';
    $precio_final = (float)($params['precio_final'] ?? 0);
    $precio_final_op = $params['precio_final_op'] ?? '>=';
    $categorias = $params['categorias'] ?? [];
    $filtro_imagen = $params['filtro_imagen'] ?? 'todos';
    $mostrar_excepciones = $params['mostrar_excepciones'] ?? 'todos';
    $mostrar_imperfectos = $params['mostrar_imperfectos'] ?? 'todos';
    
    $filtered = array_filter($combinations, function($combo) use (
        $search, $show_type, $bodegas, $almacenes, $qty, $qty_op, 
        $precio, $precio_op, $descuento, $descuento_op, $precio_final, 
        $precio_final_op, $categorias, $filtro_imagen, $mostrar_excepciones,
        $mostrar_imperfectos
    ) {
        // Filtro por tipo
        if ($show_type === 'solo_lmats' && $combo['type'] !== 'lmat') return false;
        if ($show_type === 'solo_articulos' && $combo['type'] !== 'articulo') return false;
        if ($show_type === 'solo_imperfectos' && $combo['type'] !== 'imperfecto') return false;
        
        // Filtro de excepciones
        if ($mostrar_excepciones === 'solo_excepciones' && empty($combo['es_excepcion'])) return false;
        if ($mostrar_excepciones === 'sin_excepciones' && !empty($combo['es_excepcion'])) return false;
        
        // Filtro de imperfectos
        if ($mostrar_imperfectos === 'solo_imperfectos' && empty($combo['es_imperfecto'])) return false;
        if ($mostrar_imperfectos === 'sin_imperfectos' && !empty($combo['es_imperfecto'])) return false;
        
        // Búsqueda
        if ($search) {
            $search_lower = strtolower($search);
            $found = false;
            
            // Buscar en múltiples campos
            if (stripos($combo['lmat'], $search) !== false) $found = true;
            if (stripos($combo['nombre'] ?? '', $search) !== false) $found = true;
            if (isset($combo['codigo_unico']) && stripos($combo['codigo_unico'], $search) !== false) $found = true;
            if (isset($combo['etiqueta']) && stripos($combo['etiqueta'], $search) !== false) $found = true;
            
            if (!$found) return false;
        }
        
        // Filtros de ubicación
        if (!empty($bodegas) && !in_array($combo['bodega'], $bodegas)) return false;
        if (!empty($almacenes) && !in_array($combo['almacen'], $almacenes)) return false;
        
        // Filtro de categorías
        if (!empty($categorias)) {
            $cat_match = false;
            if (in_array($combo['categoria'] ?? '', $categorias)) $cat_match = true;
            if (in_array($combo['categoria1'] ?? '', $categorias)) $cat_match = true;
            if (in_array($combo['categoria2'] ?? '', $categorias)) $cat_match = true;
            if (!$cat_match) return false;
        }
        
        // Filtro de imagen
        if ($filtro_imagen === 'con_foto' && empty($combo['url_imagen'])) return false;
        if ($filtro_imagen === 'sin_foto' && !empty($combo['url_imagen'])) return false;
        
        // Filtros numéricos
        if ($qty > 0) {
            if ($qty_op === '>=' && $combo['units'] < $qty) return false;
            if ($qty_op === '<=' && $combo['units'] > $qty) return false;
            if ($qty_op === '=' && $combo['units'] != $qty) return false;
        }
        
        if ($precio > 0) {
            $combo_precio = (float)($combo['precio'] ?? 0);
            if ($precio_op === '>=' && $combo_precio < $precio) return false;
            if ($precio_op === '<=' && $combo_precio > $precio) return false;
            if ($precio_op === '=' && $combo_precio != $precio) return false;
        }
        
        if ($descuento > 0) {
            $combo_descuento = (int)($combo['descuento'] ?? 0);
            if ($descuento_op === '>=' && $combo_descuento < $descuento) return false;
            if ($descuento_op === '<=' && $combo_descuento > $descuento) return false;
            if ($descuento_op === '=' && $combo_descuento != $descuento) return false;
        }
        
        if ($precio_final > 0) {
            $combo_precio_final = (float)($combo['precio_final'] ?? 0);
            if ($precio_final_op === '>=' && $combo_precio_final < $precio_final) return false;
            if ($precio_final_op === '<=' && $combo_precio_final > $precio_final) return false;
            if ($precio_final_op === '=' && $combo_precio_final != $precio_final) return false;
        }
        
        return true;
    });

    // Ordenamiento
    usort($filtered, function($a, $b) use ($sort) {
        // Priorizar productos sin foto si se solicita
        if ($sort === 'sin_foto_primero') {
            $a_has_image = !empty($a['url_imagen']);
            $b_has_image = !empty($b['url_imagen']);
            if ($a_has_image !== $b_has_image) {
                return $a_has_image <=> $b_has_image;
            }
        }

        switch ($sort) {
            case 'price_desc': 
                return ($b['precio_final'] ?? 0) <=> ($a['precio_final'] ?? 0);
            case 'price_asc': 
                return ($a['precio_final'] ?? 0) <=> ($b['precio_final'] ?? 0);
            case 'qty_desc':
                return ($b['units'] ?? 0) <=> ($a['units'] ?? 0);
            case 'qty_asc':
                return ($a['units'] ?? 0) <=> ($b['units'] ?? 0);
            case 'alpha_desc': 
                return strcmp($b['lmat'], $a['lmat']);
            case 'imperfectos_primero':
                // Priorizar imperfectos
                $a_imp = isset($a['es_imperfecto']) ? 0 : 1;
                $b_imp = isset($b['es_imperfecto']) ? 0 : 1;
                if ($a_imp !== $b_imp) return $a_imp <=> $b_imp;
                return strcmp($a['lmat'], $b['lmat']);
            case 'excepciones_primero':
                // Priorizar excepciones
                $a_exc = isset($a['es_excepcion']) && $a['es_excepcion'] ? 0 : 1;
                $b_exc = isset($b['es_excepcion']) && $b['es_excepcion'] ? 0 : 1;
                if ($a_exc !== $b_exc) return $a_exc <=> $b_exc;
                return strcmp($a['lmat'], $b['lmat']);
            default: 
                return strcmp($a['lmat'], $b['lmat']);
        }
    });

    return $filtered;
}

// Función auxiliar para obtener estadísticas
function get_stock_statistics($combinations) {
    $stats = [
        'total_items' => count($combinations),
        'total_units' => 0,
        'total_lmats' => 0,
        'total_articulos' => 0,
        'total_imperfectos' => 0,
        'total_excepciones' => 0,
        'categorias' => [],
        'bodegas' => [],
        'almacenes' => []
    ];
    
    foreach ($combinations as $item) {
        $stats['total_units'] += $item['units'];
        
        if ($item['type'] === 'lmat') $stats['total_lmats']++;
        if ($item['type'] === 'articulo') $stats['total_articulos']++;
        if ($item['type'] === 'imperfecto') $stats['total_imperfectos']++;
        if (!empty($item['es_excepcion'])) $stats['total_excepciones']++;
        
        // Categorías únicas
        if (!empty($item['categoria'])) {
            $stats['categorias'][$item['categoria']] = ($stats['categorias'][$item['categoria']] ?? 0) + 1;
        }
        
        // Bodegas únicas
        if (!empty($item['bodega'])) {
            $stats['bodegas'][$item['bodega']] = ($stats['bodegas'][$item['bodega']] ?? 0) + 1;
        }
        
        // Almacenes únicos
        if (!empty($item['almacen'])) {
            $stats['almacenes'][$item['almacen']] = ($stats['almacenes'][$item['almacen']] ?? 0) + 1;
        }
    }
    
    return $stats;
}
?>